const { expect } = require("chai");
const { ethers } = require("hardhat");

describe("ObitixCustody", function () {
  let obitixCustody;
  let owner;
  let addr1;
  let addr2;
  let addr3;

  beforeEach(async function () {
    // Get signers
    [owner, addr1, addr2, addr3] = await ethers.getSigners();

    // Deploy contract
    const ObitixCustody = await ethers.getContractFactory("ObitixCustody");
    obitixCustody = await ObitixCustody.deploy();
  });

  describe("Deployment", function () {
    it("Should set the right owner", async function () {
      expect(await obitixCustody.owner()).to.equal(owner.address);
    });

    it("Should start with zero organizations", async function () {
      expect(await obitixCustody.getOrganizationCount()).to.equal(0);
    });

    it("Should start with zero transports", async function () {
      expect(await obitixCustody.getTransportCount()).to.equal(0);
    });
  });

  describe("Organization Management", function () {
    it("Should register a new organization", async function () {
      const orgName = "Test Funeral Home";
      const orgAddress = addr1.address;
      const orgType = "funeral_home";

      await obitixCustody.registerOrganization(orgName, orgAddress, orgType);

      const org = await obitixCustody.getOrganization(orgAddress);
      expect(org.name).to.equal(orgName);
      expect(org.organizationAddress).to.equal(orgAddress);
      expect(org.orgType).to.equal(orgType);
      expect(org.isActive).to.be.true;
    });

    it("Should not allow duplicate organization registration", async function () {
      const orgName = "Test Funeral Home";
      const orgAddress = addr1.address;
      const orgType = "funeral_home";

      await obitixCustody.registerOrganization(orgName, orgAddress, orgType);

      await expect(
        obitixCustody.registerOrganization(orgName, orgAddress, orgType)
      ).to.be.revertedWith("Organization already exists");
    });

    it("Should allow owner to deactivate organization", async function () {
      const orgName = "Test Funeral Home";
      const orgAddress = addr1.address;
      const orgType = "funeral_home";

      await obitixCustody.registerOrganization(orgName, orgAddress, orgType);
      await obitixCustody.deactivateOrganization(orgAddress);

      const org = await obitixCustody.getOrganization(orgAddress);
      expect(org.isActive).to.be.false;
    });

    it("Should not allow non-owner to deactivate organization", async function () {
      const orgName = "Test Funeral Home";
      const orgAddress = addr1.address;
      const orgType = "funeral_home";

      await obitixCustody.registerOrganization(orgName, orgAddress, orgType);

      await expect(
        obitixCustody.connect(addr1).deactivateOrganization(orgAddress)
      ).to.be.revertedWith("Ownable: caller is not the owner");
    });
  });

  describe("Transport Management", function () {
    beforeEach(async function () {
      // Register organization first
      await obitixCustody.registerOrganization(
        "Test Funeral Home",
        addr1.address,
        "funeral_home"
      );
    });

    it("Should create a new transport", async function () {
      const transportId = "T001";
      const deceasedName = "John Doe";
      const pickupLocation = "123 Main St";
      const destinationLocation = "456 Cemetery Rd";
      const scheduledTime = Math.floor(Date.now() / 1000) + 3600; // 1 hour from now

      await obitixCustody.createTransport(
        transportId,
        deceasedName,
        pickupLocation,
        destinationLocation,
        scheduledTime,
        addr1.address
      );

      const transport = await obitixCustody.getTransport(transportId);
      expect(transport.transportId).to.equal(transportId);
      expect(transport.deceasedName).to.equal(deceasedName);
      expect(transport.pickupLocation).to.equal(pickupLocation);
      expect(transport.destinationLocation).to.equal(destinationLocation);
      expect(transport.scheduledTime).to.equal(scheduledTime);
      expect(transport.organizationAddress).to.equal(addr1.address);
      expect(transport.status).to.equal("scheduled");
    });

    it("Should not allow duplicate transport ID", async function () {
      const transportId = "T001";
      const deceasedName = "John Doe";
      const pickupLocation = "123 Main St";
      const destinationLocation = "456 Cemetery Rd";
      const scheduledTime = Math.floor(Date.now() / 1000) + 3600;

      await obitixCustody.createTransport(
        transportId,
        deceasedName,
        pickupLocation,
        destinationLocation,
        scheduledTime,
        addr1.address
      );

      await expect(
        obitixCustody.createTransport(
          transportId,
          "Jane Doe",
          pickupLocation,
          destinationLocation,
          scheduledTime,
          addr1.address
        )
      ).to.be.revertedWith("Transport already exists");
    });

    it("Should not allow transport creation for inactive organization", async function () {
      // Deactivate organization
      await obitixCustody.deactivateOrganization(addr1.address);

      const transportId = "T001";
      const deceasedName = "John Doe";
      const pickupLocation = "123 Main St";
      const destinationLocation = "456 Cemetery Rd";
      const scheduledTime = Math.floor(Date.now() / 1000) + 3600;

      await expect(
        obitixCustody.createTransport(
          transportId,
          deceasedName,
          pickupLocation,
          destinationLocation,
          scheduledTime,
          addr1.address
        )
      ).to.be.revertedWith("Organization is not active");
    });
  });

  describe("Custody Events", function () {
    beforeEach(async function () {
      // Register organization and create transport
      await obitixCustody.registerOrganization(
        "Test Funeral Home",
        addr1.address,
        "funeral_home"
      );

      await obitixCustody.createTransport(
        "T001",
        "John Doe",
        "123 Main St",
        "456 Cemetery Rd",
        Math.floor(Date.now() / 1000) + 3600,
        addr1.address
      );
    });

    it("Should record a custody event", async function () {
      const transportId = "T001";
      const eventType = "pickup";
      const location = "123 Main St";
      const timestamp = Math.floor(Date.now() / 1000);
      const technicianAddress = addr2.address;
      const notes = "Picked up from hospital";

      await obitixCustody.recordCustodyEvent(
        transportId,
        eventType,
        location,
        timestamp,
        technicianAddress,
        notes
      );

      const events = await obitixCustody.getCustodyEvents(transportId);
      expect(events.length).to.equal(1);
      expect(events[0].eventType).to.equal(eventType);
      expect(events[0].location).to.equal(location);
      expect(events[0].timestamp).to.equal(timestamp);
      expect(events[0].technicianAddress).to.equal(technicianAddress);
      expect(events[0].notes).to.equal(notes);
    });

    it("Should not allow custody event for non-existent transport", async function () {
      const transportId = "NONEXISTENT";
      const eventType = "pickup";
      const location = "123 Main St";
      const timestamp = Math.floor(Date.now() / 1000);
      const technicianAddress = addr2.address;
      const notes = "Picked up from hospital";

      await expect(
        obitixCustody.recordCustodyEvent(
          transportId,
          eventType,
          location,
          timestamp,
          technicianAddress,
          notes
        )
      ).to.be.revertedWith("Transport does not exist");
    });

    it("Should emit CustodyEventRecorded event", async function () {
      const transportId = "T001";
      const eventType = "pickup";
      const location = "123 Main St";
      const timestamp = Math.floor(Date.now() / 1000);
      const technicianAddress = addr2.address;
      const notes = "Picked up from hospital";

      await expect(
        obitixCustody.recordCustodyEvent(
          transportId,
          eventType,
          location,
          timestamp,
          technicianAddress,
          notes
        )
      ).to.emit(obitixCustody, "CustodyEventRecorded")
        .withArgs(transportId, eventType, location, timestamp, technicianAddress);
    });
  });

  describe("Transport Status Updates", function () {
    beforeEach(async function () {
      // Register organization and create transport
      await obitixCustody.registerOrganization(
        "Test Funeral Home",
        addr1.address,
        "funeral_home"
      );

      await obitixCustody.createTransport(
        "T001",
        "John Doe",
        "123 Main St",
        "456 Cemetery Rd",
        Math.floor(Date.now() / 1000) + 3600,
        addr1.address
      );
    });

    it("Should update transport status", async function () {
      const transportId = "T001";
      const newStatus = "in_progress";

      await obitixCustody.updateTransportStatus(transportId, newStatus);

      const transport = await obitixCustody.getTransport(transportId);
      expect(transport.status).to.equal(newStatus);
    });

    it("Should not allow status update for non-existent transport", async function () {
      const transportId = "NONEXISTENT";
      const newStatus = "in_progress";

      await expect(
        obitixCustody.updateTransportStatus(transportId, newStatus)
      ).to.be.revertedWith("Transport does not exist");
    });

    it("Should emit TransportStatusUpdated event", async function () {
      const transportId = "T001";
      const newStatus = "in_progress";

      await expect(
        obitixCustody.updateTransportStatus(transportId, newStatus)
      ).to.emit(obitixCustody, "TransportStatusUpdated")
        .withArgs(transportId, newStatus);
    });
  });

  describe("Access Control", function () {
    it("Should allow only owner to pause contract", async function () {
      await obitixCustody.pause();

      // Try to create transport while paused
      await obitixCustody.registerOrganization(
        "Test Funeral Home",
        addr1.address,
        "funeral_home"
      );

      await expect(
        obitixCustody.createTransport(
          "T001",
          "John Doe",
          "123 Main St",
          "456 Cemetery Rd",
          Math.floor(Date.now() / 1000) + 3600,
          addr1.address
        )
      ).to.be.revertedWith("Pausable: paused");
    });

    it("Should not allow non-owner to pause contract", async function () {
      await expect(
        obitixCustody.connect(addr1).pause()
      ).to.be.revertedWith("Ownable: caller is not the owner");
    });

    it("Should allow owner to unpause contract", async function () {
      await obitixCustody.pause();
      await obitixCustody.unpause();

      // Should be able to create transport after unpausing
      await obitixCustody.registerOrganization(
        "Test Funeral Home",
        addr1.address,
        "funeral_home"
      );

      await obitixCustody.createTransport(
        "T001",
        "John Doe",
        "123 Main St",
        "456 Cemetery Rd",
        Math.floor(Date.now() / 1000) + 3600,
        addr1.address
      );

      const transport = await obitixCustody.getTransport("T001");
      expect(transport.transportId).to.equal("T001");
    });
  });

  describe("Query Functions", function () {
    beforeEach(async function () {
      // Register multiple organizations
      await obitixCustody.registerOrganization("Org1", addr1.address, "funeral_home");
      await obitixCustody.registerOrganization("Org2", addr2.address, "crematory");
      await obitixCustody.registerOrganization("Org3", addr3.address, "cemetery");

      // Create multiple transports
      await obitixCustody.createTransport("T001", "John Doe", "Loc1", "Dest1", Math.floor(Date.now() / 1000) + 3600, addr1.address);
      await obitixCustody.createTransport("T002", "Jane Doe", "Loc2", "Dest2", Math.floor(Date.now() / 1000) + 7200, addr2.address);
      await obitixCustody.createTransport("T003", "Bob Smith", "Loc3", "Dest3", Math.floor(Date.now() / 1000) + 10800, addr1.address);
    });

    it("Should return correct organization count", async function () {
      expect(await obitixCustody.getOrganizationCount()).to.equal(3);
    });

    it("Should return correct transport count", async function () {
      expect(await obitixCustody.getTransportCount()).to.equal(3);
    });

    it("Should return transports for organization", async function () {
      const transports = await obitixCustody.getTransportsByOrganization(addr1.address);
      expect(transports.length).to.equal(2);
      expect(transports[0]).to.equal("T001");
      expect(transports[1]).to.equal("T003");
    });

    it("Should return empty array for organization with no transports", async function () {
      // Register new organization without transports
      await obitixCustody.registerOrganization("Org4", ethers.Wallet.createRandom().address, "funeral_home");
      
      const transports = await obitixCustody.getTransportsByOrganization(ethers.Wallet.createRandom().address);
      expect(transports.length).to.equal(0);
    });
  });
});
