const fs = require('fs');
const path = require('path');
const { 
  uploadToIPFS, 
  uploadBufferToIPFS, 
  getFromIPFS, 
  uploadMetadataToIPFS,
  getMetadataFromIPFS,
  listPinnedFiles,
  getFileInfo,
  getGatewayURL,
  isValidIPFSHash
} = require('./src/services/ipfs');

require('dotenv').config();

// Test configuration
const TEST_FILE_PATH = path.join(__dirname, 'test-file.txt');
const TEST_METADATA = {
  service: 'obitix',
  type: 'test',
  version: '1.0.0'
};

// Create a test file
function createTestFile() {
  const content = `This is a test file for Obitix IPFS storage.
Created at: ${new Date().toISOString()}
Service: Obitix
Storage: Pinata IPFS (No AWS Required!)`;
  
  fs.writeFileSync(TEST_FILE_PATH, content);
  console.log('✅ Test file created:', TEST_FILE_PATH);
}

// Test file upload
async function testFileUpload() {
  console.log('\n🧪 Testing File Upload to Pinata IPFS...');
  
  try {
    const result = await uploadToIPFS(TEST_FILE_PATH, 'test-file.txt', TEST_METADATA);
    console.log('✅ File uploaded successfully!');
    console.log('   Hash:', result.hash);
    console.log('   Size:', result.size, 'bytes');
    console.log('   Gateway URL:', result.gatewayUrl);
    return result.hash;
  } catch (error) {
    console.error('❌ File upload failed:', error.message);
    return null;
  }
}

// Test buffer upload
async function testBufferUpload() {
  console.log('\n🧪 Testing Buffer Upload to Pinata IPFS...');
  
  try {
    const testData = {
      message: 'Hello from Obitix!',
      timestamp: new Date().toISOString(),
      service: 'obitix',
      storage: 'pinata-ipfs'
    };
    
    const buffer = Buffer.from(JSON.stringify(testData, null, 2));
    const result = await uploadBufferToIPFS(buffer, 'test-data.json', {
      type: 'json',
      contentType: 'application/json'
    });
    
    console.log('✅ Buffer uploaded successfully!');
    console.log('   Hash:', result.hash);
    console.log('   Size:', result.size, 'bytes');
    console.log('   Gateway URL:', result.gatewayUrl);
    return result.hash;
  } catch (error) {
    console.error('❌ Buffer upload failed:', error.message);
    return null;
  }
}

// Test metadata upload
async function testMetadataUpload() {
  console.log('\n🧪 Testing Metadata Upload to Pinata IPFS...');
  
  try {
    const metadata = {
      name: 'Test Document',
      description: 'A test document for Obitix',
      author: 'Obitix System',
      version: '1.0.0',
      tags: ['test', 'obitix', 'ipfs'],
      created: new Date().toISOString(),
      storage: {
        provider: 'pinata',
        type: 'ipfs',
        aws_required: false
      }
    };
    
    const result = await uploadMetadataToIPFS(metadata, 'test-metadata.json');
    console.log('✅ Metadata uploaded successfully!');
    console.log('   Hash:', result.hash);
    console.log('   Size:', result.size, 'bytes');
    console.log('   Gateway URL:', result.gatewayUrl);
    return result.hash;
  } catch (error) {
    console.error('❌ Metadata upload failed:', error.message);
    return null;
  }
}

// Test file retrieval
async function testFileRetrieval(hash) {
  console.log('\n🧪 Testing File Retrieval from Pinata IPFS...');
  
  try {
    const buffer = await getFromIPFS(hash);
    const content = buffer.toString();
    console.log('✅ File retrieved successfully!');
    console.log('   Content length:', content.length, 'characters');
    console.log('   Content preview:', content.substring(0, 100) + '...');
    return true;
  } catch (error) {
    console.error('❌ File retrieval failed:', error.message);
    return false;
  }
}

// Test metadata retrieval
async function testMetadataRetrieval(hash) {
  console.log('\n🧪 Testing Metadata Retrieval from Pinata IPFS...');
  
  try {
    const metadata = await getMetadataFromIPFS(hash);
    console.log('✅ Metadata retrieved successfully!');
    console.log('   Name:', metadata.name);
    console.log('   Description:', metadata.description);
    console.log('   Author:', metadata.author);
    console.log('   Storage provider:', metadata.storage.provider);
    return true;
  } catch (error) {
    console.error('❌ Metadata retrieval failed:', error.message);
    return false;
  }
}

// Test file info
async function testFileInfo(hash) {
  console.log('\n🧪 Testing File Info from Pinata...');
  
  try {
    const info = await getFileInfo(hash);
    console.log('✅ File info retrieved successfully!');
    console.log('   Hash:', info.hash);
    console.log('   Size:', info.size, 'bytes');
    console.log('   Name:', info.name);
    console.log('   Timestamp:', info.timestamp);
    console.log('   Gateway URL:', info.gatewayUrl);
    return true;
  } catch (error) {
    console.error('❌ File info retrieval failed:', error.message);
    return false;
  }
}

// Test hash validation
function testHashValidation(hash) {
  console.log('\n🧪 Testing Hash Validation...');
  
  try {
    const isValid = isValidIPFSHash(hash);
    console.log('✅ Hash validation completed!');
    console.log('   Hash:', hash);
    console.log('   Is valid:', isValid);
    return isValid;
  } catch (error) {
    console.error('❌ Hash validation failed:', error.message);
    return false;
  }
}

// Test gateway URL generation
function testGatewayURL(hash) {
  console.log('\n🧪 Testing Gateway URL Generation...');
  
  try {
    const gatewayUrl = getGatewayURL(hash);
    console.log('✅ Gateway URL generated!');
    console.log('   Hash:', hash);
    console.log('   Gateway URL:', gatewayUrl);
    return gatewayUrl;
  } catch (error) {
    console.error('❌ Gateway URL generation failed:', error.message);
    return null;
  }
}

// Test listing pinned files
async function testListPinnedFiles() {
  console.log('\n🧪 Testing List Pinned Files from Pinata...');
  
  try {
    const files = await listPinnedFiles({ limit: 5 });
    console.log('✅ Pinned files listed successfully!');
    console.log('   Total files found:', files.length);
    
    files.forEach((file, index) => {
      console.log(`   ${index + 1}. ${file.name} (${file.hash})`);
    });
    
    return files.length;
  } catch (error) {
    console.error('❌ List pinned files failed:', error.message);
    return 0;
  }
}

// Main test function
async function runTests() {
  console.log('🚀 Starting Pinata IPFS Tests (No AWS Required!)');
  console.log('=' .repeat(60));
  
  // Check environment
  if (!process.env.PINATA_API_KEY || (!process.env.PINATA_SECRET_KEY && !process.env.PINATA_SECRET_API_KEY)) {
    console.error('❌ Pinata API credentials not configured!');
    console.log('Please set PINATA_API_KEY and PINATA_SECRET_KEY in your .env file');
    return;
  }
  
  // Create test file
  createTestFile();
  
  // Run tests
  const fileHash = await testFileUpload();
  const bufferHash = await testBufferUpload();
  const metadataHash = await testMetadataUpload();
  
  if (fileHash) {
    await testFileRetrieval(fileHash);
    await testFileInfo(fileHash);
    testHashValidation(fileHash);
    testGatewayURL(fileHash);
  }
  
  if (metadataHash) {
    await testMetadataRetrieval(metadataHash);
  }
  
  await testListPinnedFiles();
  
  // Cleanup
  if (fs.existsSync(TEST_FILE_PATH)) {
    fs.unlinkSync(TEST_FILE_PATH);
    console.log('\n🧹 Test file cleaned up');
  }
  
  console.log('\n🎉 All Pinata IPFS tests completed!');
  console.log('✅ No AWS dependencies required');
  console.log('✅ All files stored on IPFS via Pinata');
  console.log('✅ Cost-effective and decentralized storage');
}

// Run tests if this file is executed directly
if (require.main === module) {
  runTests().catch(console.error);
}

module.exports = {
  runTests,
  testFileUpload,
  testBufferUpload,
  testMetadataUpload,
  testFileRetrieval,
  testMetadataRetrieval,
  testFileInfo,
  testHashValidation,
  testGatewayURL,
  testListPinnedFiles
};
