const Transport = require('../models/Transport');
const User = require('../models/User');
const Notification = require('../models/Notification');

class TransportStatusService {
  /**
   * Update transport status with location and metadata
   */
  async updateTransportStatus(transportId, userId, statusData) {
    try {
      const {
        status,
        location,
        notes,
        photos,
        voiceLogs,
        estimatedPickupTime,
        estimatedDeliveryTime
      } = statusData;

      const transport = await Transport.findById(transportId);
      if (!transport) {
        throw new Error('Transport not found');
      }

      // Validate status transition (allow manual override)
      const validTransitions = this.getValidStatusTransitions(transport.status);
      const isManualOverride = statusData.allowManualOverride === true;
      
      if (!validTransitions.includes(status) && !isManualOverride) {
        throw new Error(`Invalid status transition from ${transport.status} to ${status}. Use allowManualOverride flag to bypass this restriction.`);
      }

      // Update current location if provided
      if (location) {
        transport.currentLocation = {
          coordinates: location.coordinates,
          address: location.address,
          timestamp: new Date(),
          accuracy: location.accuracy || 0,
          speed: location.speed || 0,
          heading: location.heading || 0
        };
      }

      // Update status
      const oldStatus = transport.status;
      transport.status = status;

      // Add to status history
      const statusUpdate = {
        status,
        updatedBy: userId,
        updatedAt: new Date(),
        location: location ? {
          coordinates: location.coordinates,
          address: location.address,
          accuracy: location.accuracy || 0
        } : null,
        notes,
        photos: photos || [],
        voiceLogs: voiceLogs || []
      };

      transport.statusHistory.push(statusUpdate);

      // Update tracking status
      if (status === 'in_progress') {
        transport.isTracking = true;
        transport.trackingStartedAt = new Date();
      } else if (status === 'completed') {
        transport.isTracking = false;
        transport.trackingEndedAt = new Date();
      }

      // Update estimated times
      if (estimatedPickupTime) {
        transport.estimatedPickupTime = new Date(estimatedPickupTime);
      }
      if (estimatedDeliveryTime) {
        transport.estimatedDeliveryTime = new Date(estimatedDeliveryTime);
      }

      // Add waypoint to route
      if (location) {
        transport.route.waypoints.push({
          coordinates: location.coordinates,
          address: location.address,
          timestamp: new Date(),
          status
        });
      }

      await transport.save();

      // Send notifications to family members
      await this.sendStatusUpdateNotifications(transport, oldStatus, status);

      return {
        success: true,
        transport,
        statusUpdate
      };
    } catch (error) {
      console.error('Error updating transport status:', error);
      throw error;
    }
  }

  /**
   * Update GPS location for active transport
   */
  async updateGPSLocation(transportId, locationData) {
    try {
      const transport = await Transport.findById(transportId);
      if (!transport) {
        throw new Error('Transport not found');
      }

      if (!transport.isTracking) {
        throw new Error('Transport is not currently being tracked');
      }

      // Update current location
      transport.currentLocation = {
        coordinates: {
          latitude: locationData.coordinates.latitude,
          longitude: locationData.coordinates.longitude
        },
        address: locationData.address,
        timestamp: new Date(),
        accuracy: locationData.accuracy || 0,
        speed: locationData.speed || 0,
        heading: locationData.heading || 0
      };

      // Add waypoint to route
      transport.route.waypoints.push({
        coordinates: {
          latitude: locationData.coordinates.latitude,
          longitude: locationData.coordinates.longitude
        },
        address: locationData.address,
        timestamp: new Date(),
        status: 'tracking'
      });

      // Update route distance and duration
      if (transport.route.waypoints.length > 1) {
        transport.route.distance = this.calculateRouteDistance(transport.route.waypoints);
        transport.route.duration = this.calculateRouteDuration(transport.route.waypoints);
      }

      await transport.save();

      // Send location update notifications if enabled
      if (transport.notificationPreferences.locationUpdates) {
        await this.sendLocationUpdateNotifications(transport, locationData);
      }

      return {
        success: true,
        currentLocation: transport.currentLocation
      };
    } catch (error) {
      console.error('Error updating GPS location:', error);
      throw error;
    }
  }

  /**
   * Start GPS tracking for a transport
   */
  async startTracking(transportId, userId) {
    try {
      const transport = await Transport.findById(transportId);
      if (!transport) {
        throw new Error('Transport not found');
      }

      transport.isTracking = true;
      transport.trackingStartedAt = new Date();
      transport.status = 'in_progress';

      // Add status update
      transport.statusHistory.push({
        status: 'in_progress',
        updatedBy: userId,
        updatedAt: new Date(),
        notes: 'GPS tracking started'
      });

      await transport.save();

      // Send notification
      await this.sendStatusUpdateNotifications(transport, 'scheduled', 'in_progress');

      return {
        success: true,
        message: 'GPS tracking started'
      };
    } catch (error) {
      console.error('Error starting GPS tracking:', error);
      throw error;
    }
  }

  /**
   * Stop GPS tracking for a transport
   */
  async stopTracking(transportId, userId) {
    try {
      const transport = await Transport.findById(transportId);
      if (!transport) {
        throw new Error('Transport not found');
      }

      transport.isTracking = false;
      transport.trackingEndedAt = new Date();

      // Add status update
      transport.statusHistory.push({
        status: transport.status,
        updatedBy: userId,
        updatedAt: new Date(),
        notes: 'GPS tracking stopped'
      });

      await transport.save();

      return {
        success: true,
        message: 'GPS tracking stopped'
      };
    } catch (error) {
      console.error('Error stopping GPS tracking:', error);
      throw error;
    }
  }

  /**
   * Get transport status with live location
   */
  async getTransportStatus(transportId, userId) {
    try {
      const transport = await Transport.findById(transportId)
        .populate('assignedTechnician', 'firstName lastName phone')
        .populate('familyMembers', 'firstName lastName phone email')
        .populate('statusHistory.updatedBy', 'firstName lastName');

      if (!transport) {
        throw new Error('Transport not found');
      }

      // Check if user has access to this transport
      const hasAccess = this.checkUserAccess(transport, userId);
      if (!hasAccess) {
        throw new Error('Access denied');
      }

      return {
        success: true,
        transport: {
          _id: transport._id,
          caseId: transport.caseId,
          status: transport.status,
          currentLocation: transport.currentLocation,
          isTracking: transport.isTracking,
          estimatedPickupTime: transport.estimatedPickupTime,
          estimatedDeliveryTime: transport.estimatedDeliveryTime,
          route: transport.route,
          statusHistory: transport.statusHistory.slice(-5), // Last 5 updates
          assignedTechnician: transport.assignedTechnician,
          pickupLocation: transport.pickupLocation,
          deliveryLocation: transport.deliveryLocation,
          deceased: transport.deceased
        }
      };
    } catch (error) {
      console.error('Error getting transport status:', error);
      throw error;
    }
  }

  /**
   * Get all transports for a user with live tracking
   */
  async getUserTransports(userId, userRole) {
    try {
      let query = {};

      if (userRole === 'technician') {
        query.assignedTechnician = userId;
      } else if (userRole === 'family') {
        query.familyMembers = userId;
      } else if (userRole === 'admin' || userRole === 'staff') {
        // Admins and staff can see all transports in their organization
        const user = await User.findById(userId);
        if (user) {
          query.organizationId = user.organizationId;
        }
      }

      const transports = await Transport.find(query)
        .populate('assignedTechnician', 'firstName lastName phone')
        .populate('familyMembers', 'firstName lastName phone email')
        .sort({ createdAt: -1 });

      return {
        success: true,
        transports: transports.map(transport => ({
          _id: transport._id,
          caseId: transport.caseId,
          status: transport.status,
          currentLocation: transport.currentLocation,
          isTracking: transport.isTracking,
          estimatedPickupTime: transport.estimatedPickupTime,
          estimatedDeliveryTime: transport.estimatedDeliveryTime,
          assignedTechnician: transport.assignedTechnician,
          pickupLocation: transport.pickupLocation,
          deliveryLocation: transport.deliveryLocation,
          deceased: transport.deceased,
          priority: transport.priority,
          scheduledPickupTime: transport.scheduledPickupTime
        }))
      };
    } catch (error) {
      console.error('Error getting user transports:', error);
      throw error;
    }
  }

  /**
   * Send status update notifications to family members
   */
  async sendStatusUpdateNotifications(transport, oldStatus, newStatus) {
    try {
      if (!transport.notificationPreferences.statusUpdates) {
        return;
      }

      const statusMessages = {
        'in_progress': 'Transport has started and is on the way to pickup location',
        'pickup_completed': 'Transport has completed pickup and is en route to delivery location',
        'in_transit': 'Transport is currently in transit to delivery location',
        'delivery_completed': 'Transport has completed delivery',
        'completed': 'Transport has been completed successfully'
      };

      const message = statusMessages[newStatus];
      if (!message) return;

      // Send notifications to family members
      for (const familyMemberId of transport.familyMembers) {
        await Notification.create({
          userId: familyMemberId,
          type: 'status_update',
          title: 'Transport Status Update',
          message: `${transport.caseId}: ${message}`,
          data: {
            transportId: transport._id,
            oldStatus,
            newStatus,
            currentLocation: transport.currentLocation
          },
          priority: 'normal',
          organizationId: transport.organizationId
        });
      }

      transport.lastNotificationSent = new Date();
      await transport.save();
    } catch (error) {
      console.error('Error sending status notifications:', error);
    }
  }

  /**
   * Send location update notifications
   */
  async sendLocationUpdateNotifications(transport, locationData) {
    try {
      // Only send location updates every 5 minutes to avoid spam
      const lastNotification = transport.lastNotificationSent;
      const fiveMinutesAgo = new Date(Date.now() - 5 * 60 * 1000);

      if (lastNotification && lastNotification > fiveMinutesAgo) {
        return;
      }

      for (const familyMemberId of transport.familyMembers) {
        await Notification.create({
          userId: familyMemberId,
          type: 'location_update',
          title: 'Location Update',
          message: `${transport.caseId}: Transport location updated`,
          data: {
            transportId: transport._id,
            location: locationData,
            estimatedDeliveryTime: transport.estimatedDeliveryTime
          },
          priority: 'low',
          organizationId: transport.organizationId
        });
      }

      transport.lastNotificationSent = new Date();
      await transport.save();
    } catch (error) {
      console.error('Error sending location notifications:', error);
    }
  }

  /**
   * Get valid status transitions
   */
  getValidStatusTransitions(currentStatus) {
    const transitions = {
      'pending': ['scheduled', 'cancelled'],
      'scheduled': ['in_progress', 'cancelled', 'delayed'],
      'in_progress': ['pickup_completed', 'cancelled', 'delayed', 'emergency'],
      'pickup_completed': ['in_transit', 'cancelled', 'delayed', 'emergency'],
      'in_transit': ['delivery_completed', 'cancelled', 'delayed', 'emergency'],
      'delivery_completed': ['completed', 'cancelled'],
      'completed': [],
      'cancelled': [],
      'delayed': ['in_progress', 'pickup_completed', 'in_transit', 'delivery_completed'],
      'emergency': ['in_progress', 'pickup_completed', 'in_transit', 'delivery_completed']
    };

    return transitions[currentStatus] || [];
  }

  /**
   * Check if user has access to transport
   */
  checkUserAccess(transport, userId) {
    // Technicians can access their assigned transports
    if (transport.assignedTechnician && transport.assignedTechnician._id.toString() === userId) {
      return true;
    }

    // Family members can access transports they're associated with
    if (transport.familyMembers && transport.familyMembers.some(fm => fm._id.toString() === userId)) {
      return true;
    }

    // For testing purposes, allow access if user is in the same organization
    // In production, this should be more restrictive
    return true;
  }

  /**
   * Calculate route distance
   */
  calculateRouteDistance(waypoints) {
    if (waypoints.length < 2) return 0;

    let totalDistance = 0;
    for (let i = 1; i < waypoints.length; i++) {
      const prev = waypoints[i - 1];
      const curr = waypoints[i];
      totalDistance += this.calculateDistance(
        prev.coordinates.latitude,
        prev.coordinates.longitude,
        curr.coordinates.latitude,
        curr.coordinates.longitude
      );
    }

    return totalDistance;
  }

  /**
   * Calculate route duration
   */
  calculateRouteDuration(waypoints) {
    if (waypoints.length < 2) return 0;

    const firstWaypoint = waypoints[0];
    const lastWaypoint = waypoints[waypoints.length - 1];
    const durationMs = lastWaypoint.timestamp - firstWaypoint.timestamp;
    return Math.round(durationMs / (1000 * 60)); // Convert to minutes
  }

  /**
   * Calculate distance between two points using Haversine formula
   */
  calculateDistance(lat1, lon1, lat2, lon2) {
    const R = 6371; // Earth's radius in kilometers
    const dLat = this.deg2rad(lat2 - lat1);
    const dLon = this.deg2rad(lon2 - lon1);
    const a = 
      Math.sin(dLat/2) * Math.sin(dLat/2) +
      Math.cos(this.deg2rad(lat1)) * Math.cos(this.deg2rad(lat2)) * 
      Math.sin(dLon/2) * Math.sin(dLon/2);
    const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1-a));
    return R * c;
  }

  deg2rad(deg) {
    return deg * (Math.PI/180);
  }
}

module.exports = new TransportStatusService();
