const fs = require('fs');
const path = require('path');
const PinataSDK = require('@pinata/sdk');
require('dotenv').config();

// Pinata configuration
const pinataConfig = {
  apiKey: process.env.PINATA_API_KEY,
  secretApiKey: process.env.PINATA_SECRET_KEY || process.env.PINATA_SECRET_API_KEY,
  gateway: process.env.PINATA_GATEWAY || 'https://gateway.pinata.cloud',
  jwtToken: process.env.PINATA_JWT_TOKEN || process.env.PINATA_JWT_SECRET // Support both variable names
};

console.log('Pinata Config Check:', {
  apiKey: pinataConfig.apiKey ? 'SET' : 'NOT SET',
  secretApiKey: pinataConfig.secretApiKey ? 'SET' : 'NOT SET',
  gateway: pinataConfig.gateway,
  envVars: {
    PINATA_API_KEY: process.env.PINATA_API_KEY ? 'SET' : 'NOT SET',
    PINATA_SECRET_KEY: process.env.PINATA_SECRET_KEY ? 'SET' : 'NOT SET',
    PINATA_SECRET_API_KEY: process.env.PINATA_SECRET_API_KEY ? 'SET' : 'NOT SET'
  }
});



// Upload file to Pinata IPFS using SDK
const uploadToIPFS = async (filePath, fileName, metadata = {}) => {
  try {
    console.log('Uploading file to Pinata IPFS:', fileName);
    
    // Initialize Pinata SDK with JWT token first, then fall back to API key
    let pinata;
    let authMethod = 'Unknown';
    
    if (pinataConfig.jwtToken) {
      console.log('Using JWT authentication...');
      authMethod = 'JWT';
      pinata = new PinataSDK({ pinataJWTKey: pinataConfig.jwtToken });
    } else if (pinataConfig.apiKey && pinataConfig.secretApiKey) {
      console.log('Using API Key authentication...');
      authMethod = 'API Key';
      pinata = new PinataSDK(pinataConfig.apiKey, pinataConfig.secretApiKey);
    } else {
      throw new Error('No Pinata authentication credentials available');
    }
    
    // Upload file using SDK
    const result = await pinata.pinFileToIPFS(fs.createReadStream(filePath), {
      pinataMetadata: {
        name: fileName,
        keyvalues: {
          service: 'obitix',
          timestamp: new Date().toISOString(),
          ...metadata
        }
      }
    });
    
    console.log(`File uploaded to Pinata IPFS (${authMethod}):`, result);

    return {
      hash: result.IpfsHash,
      size: result.PinSize,
      path: result.IpfsHash,
      gatewayUrl: getGatewayURL(result.IpfsHash)
    };
  } catch (error) {
    console.error('Error uploading to Pinata IPFS:', error);
    throw new Error(`Failed to upload file to Pinata IPFS: ${error.message}`);
  }
};

// Upload buffer to Pinata IPFS using SDK
const uploadBufferToIPFS = async (buffer, fileName, metadata = {}) => {
  try {
    console.log('Uploading buffer to Pinata IPFS:', fileName);
    
    // Initialize Pinata SDK with JWT token first, then fall back to API key
    let pinata;
    let authMethod = 'Unknown';
    
    if (pinataConfig.jwtToken) {
      console.log('Using JWT authentication...');
      authMethod = 'JWT';
      pinata = new PinataSDK({ pinataJWTKey: pinataConfig.jwtToken });
    } else if (pinataConfig.apiKey && pinataConfig.secretApiKey) {
      console.log('Using API Key authentication...');
      authMethod = 'API Key';
      pinata = new PinataSDK(pinataConfig.apiKey, pinataConfig.secretApiKey);
    } else {
      throw new Error('No Pinata authentication credentials available');
    }
    
    // Upload buffer using SDK
    const result = await pinata.pinFileToIPFS(buffer, {
      pinataMetadata: {
        name: fileName,
        keyvalues: {
          service: 'obitix',
          timestamp: new Date().toISOString(),
          ...metadata
        }
      }
    });
    
    console.log(`Buffer uploaded to Pinata IPFS (${authMethod}):`, result);

    return {
      hash: result.IpfsHash,
      size: result.PinSize,
      path: result.IpfsHash,
      gatewayUrl: getGatewayURL(result.IpfsHash)
    };
  } catch (error) {
    console.error('Error uploading buffer to Pinata IPFS:', error);
    throw new Error(`Failed to upload buffer to Pinata IPFS: ${error.message}`);
  }
};

// Get file from Pinata IPFS Gateway
const getFromIPFS = async (hash) => {
  try {
    const gatewayUrl = getGatewayURL(hash);
    const response = await fetch(gatewayUrl);

    if (!response.ok) {
      throw new Error(`Failed to fetch from IPFS gateway: ${response.status}`);
    }

    const buffer = await response.arrayBuffer();
    return Buffer.from(buffer);
  } catch (error) {
    console.error('Error getting file from Pinata IPFS:', error);
    throw new Error(`Failed to retrieve file from Pinata IPFS: ${error.message}`);
  }
};

// Pin existing hash to Pinata
const pinToPinata = async (hash, name, metadata = {}) => {
  try {
    if (!pinataConfig.apiKey || !pinataConfig.secretApiKey) {
      throw new Error('Pinata API credentials not configured');
    }

    const response = await fetch('https://api.pinata.cloud/pinning/pinByHash', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'pinata_api_key': pinataConfig.apiKey,
        'pinata_secret_api_key': pinataConfig.secretApiKey
      },
      body: JSON.stringify({
        hashToPin: hash,
        pinataMetadata: {
          name: name,
          keyvalues: {
            service: 'obitix',
            timestamp: new Date().toISOString(),
            ...metadata
          }
        }
      })
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Pinata API error: ${response.status} - ${errorText}`);
    }

    const result = await response.json();
    console.log('Hash pinned to Pinata:', result);
    
    return result;
  } catch (error) {
    console.error('Error pinning to Pinata:', error);
    throw new Error(`Failed to pin to Pinata: ${error.message}`);
  }
};

// Unpin file from Pinata
const unpinFromPinata = async (hash) => {
  try {
    if (!pinataConfig.apiKey || !pinataConfig.secretApiKey) {
      throw new Error('Pinata API credentials not configured');
    }

    const response = await fetch(`https://api.pinata.cloud/pinning/unpin/${hash}`, {
      method: 'DELETE',
      headers: {
        'pinata_api_key': pinataConfig.apiKey,
        'pinata_secret_api_key': pinataConfig.secretApiKey
      }
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`Pinata API error: ${response.status} - ${errorText}`);
    }

    const result = await response.json();
    console.log('File unpinned from Pinata:', result);
    
    return result;
  } catch (error) {
    console.error('Error unpinning from Pinata:', error);
    throw new Error(`Failed to unpin from Pinata: ${error.message}`);
  }
};

// Get Pinata IPFS gateway URL
const getGatewayURL = (hash) => {
  if (pinataConfig.gateway) {
    return `${pinataConfig.gateway}/ipfs/${hash}`;
  }
  return `https://ipfs.io/ipfs/${hash}`;
};

// Validate IPFS hash
const isValidIPFSHash = (hash) => {
  // Basic IPFS hash validation (CID v0 or v1)
  const ipfsHashRegex = /^Qm[1-9A-HJ-NP-Za-km-z]{44}$|^bafy[a-z2-7]{55}$/;
  return ipfsHashRegex.test(hash);
};

// Get file info from Pinata
const getFileInfo = async (hash) => {
  try {
    if (!pinataConfig.apiKey || !pinataConfig.secretApiKey) {
      throw new Error('Pinata API credentials not configured');
    }

    const response = await fetch(`https://api.pinata.cloud/data/pinList?hashContains=${hash}`, {
      headers: {
        'pinata_api_key': pinataConfig.apiKey,
        'pinata_secret_api_key': pinataConfig.secretApiKey
      }
    });

    if (!response.ok) {
      throw new Error(`Pinata API error: ${response.status}`);
    }

    const result = await response.json();
    const fileInfo = result.rows.find(row => row.ipfs_pin_hash === hash);

    if (!fileInfo) {
      throw new Error('File not found in Pinata');
    }

    return {
      hash: fileInfo.ipfs_pin_hash,
      size: fileInfo.size,
      name: fileInfo.metadata?.name || 'Unknown',
      timestamp: fileInfo.date_pinned,
      gatewayUrl: getGatewayURL(hash)
    };
  } catch (error) {
    console.error('Error getting file info from Pinata:', error);
    throw new Error(`Failed to get file info from Pinata: ${error.message}`);
  }
};

// Upload JSON metadata to Pinata IPFS
const uploadMetadataToIPFS = async (metadata, fileName = 'metadata.json') => {
  try {
    const metadataBuffer = Buffer.from(JSON.stringify(metadata, null, 2));
    return await uploadBufferToIPFS(metadataBuffer, fileName, {
      type: 'metadata',
      contentType: 'application/json'
    });
  } catch (error) {
    console.error('Error uploading metadata to Pinata IPFS:', error);
    throw new Error(`Failed to upload metadata to Pinata IPFS: ${error.message}`);
  }
};

// Get JSON metadata from Pinata IPFS
const getMetadataFromIPFS = async (hash) => {
  try {
    const buffer = await getFromIPFS(hash);
    return JSON.parse(buffer.toString());
  } catch (error) {
    console.error('Error getting metadata from Pinata IPFS:', error);
    throw new Error(`Failed to get metadata from Pinata IPFS: ${error.message}`);
  }
};

// List all pinned files from Pinata
const listPinnedFiles = async (filters = {}) => {
  try {
    if (!pinataConfig.apiKey || !pinataConfig.secretApiKey) {
      throw new Error('Pinata API credentials not configured');
    }

    let url = 'https://api.pinata.cloud/data/pinList?status=pinned';
    
    if (filters.hashContains) {
      url += `&hashContains=${filters.hashContains}`;
    }
    if (filters.metadataName) {
      url += `&metadata[name]=${filters.metadataName}`;
    }
    if (filters.limit) {
      url += `&pageLimit=${filters.limit}`;
    }

    const response = await fetch(url, {
      headers: {
        'pinata_api_key': pinataConfig.apiKey,
        'pinata_secret_api_key': pinataConfig.secretApiKey
      }
    });

    if (!response.ok) {
      throw new Error(`Pinata API error: ${response.status}`);
    }

    const result = await response.json();
    return result.rows.map(row => ({
      hash: row.ipfs_pin_hash,
      size: row.size,
      name: row.metadata?.name || 'Unknown',
      timestamp: row.date_pinned,
      gatewayUrl: getGatewayURL(row.ipfs_pin_hash)
    }));
  } catch (error) {
    console.error('Error listing pinned files from Pinata:', error);
    throw new Error(`Failed to list pinned files from Pinata: ${error.message}`);
  }
};

module.exports = {
  uploadToIPFS,
  uploadBufferToIPFS,
  getFromIPFS,
  pinToPinata,
  unpinFromPinata,
  getGatewayURL,
  isValidIPFSHash,
  getFileInfo,
  uploadMetadataToIPFS,
  getMetadataFromIPFS,
  listPinnedFiles
};
