const nodemailer = require('nodemailer');
const fs = require('fs').promises;
const path = require('path');

class EmailService {
  constructor() {
    this.transporter = null;
    this.templates = {};
    this.isInitialized = false;
  }

  async initialize() {
    try {
      // Check if email configuration exists
      if (!process.env.SMTP_HOST || !process.env.SMTP_USER || !process.env.SMTP_PASS) {
        console.log('Email configuration not found, email service will be disabled');
        this.isInitialized = false;
        return;
      }

      // Create transporter
      this.transporter = nodemailer.createTransport({
        host: process.env.SMTP_HOST,
        port: process.env.SMTP_PORT || 587,
        secure: process.env.SMTP_PORT === '465',
        auth: {
          user: process.env.SMTP_USER,
          pass: process.env.SMTP_PASS,
        },
      });

      // Load email templates
      await this.loadTemplates();

      // Test connection
      await this.transporter.verify();
      
      this.isInitialized = true;
      console.log('Email service initialized successfully');
    } catch (error) {
      console.error('Failed to initialize email service:', error);
      this.isInitialized = false;
      console.log('Email service will be disabled');
    }
  }

  async loadTemplates() {
    try {
      const templatesDir = path.join(__dirname, '../templates/emails');
      const templateFiles = await fs.readdir(templatesDir);

      for (const file of templateFiles) {
        if (file.endsWith('.html')) {
          const templateName = file.replace('.html', '');
          const templatePath = path.join(templatesDir, file);
          const templateContent = await fs.readFile(templatePath, 'utf8');
          this.templates[templateName] = templateContent;
        }
      }
    } catch (error) {
      console.error('Failed to load email templates:', error);
      // Use default templates if loading fails
      this.templates = this.getDefaultTemplates();
    }
  }

  getDefaultTemplates() {
    return {
      'email-verification': `
        <!DOCTYPE html>
        <html>
        <head>
          <meta charset="utf-8">
          <title>Verify Your Email - Obitix</title>
        </head>
        <body>
          <h2>Welcome to Obitix</h2>
          <p>Hello {{name}},</p>
          <p>Thank you for registering with Obitix. Please verify your email address by clicking the link below:</p>
          <a href="{{verificationUrl}}" style="background-color: #1a365d; color: white; padding: 12px 24px; text-decoration: none; border-radius: 4px;">Verify Email</a>
          <p>If you didn't create an account, please ignore this email.</p>
          <p>Best regards,<br>The Obitix Team</p>
        </body>
        </html>
      `,
      'password-reset': `
        <!DOCTYPE html>
        <html>
        <head>
          <meta charset="utf-8">
          <title>Password Reset - Obitix</title>
        </head>
        <body>
          <h2>Password Reset Request</h2>
          <p>Hello {{name}},</p>
          <p>You requested a password reset for your Obitix account. Click the link below to reset your password:</p>
          <a href="{{resetUrl}}" style="background-color: #1a365d; color: white; padding: 12px 24px; text-decoration: none; border-radius: 4px;">Reset Password</a>
          <p>This link will expire in 1 hour.</p>
          <p>If you didn't request this reset, please ignore this email.</p>
          <p>Best regards,<br>The Obitix Team</p>
        </body>
        </html>
      `,
      'transport-update': `
        <!DOCTYPE html>
        <html>
        <head>
          <meta charset="utf-8">
          <title>Transport Update - Obitix</title>
        </head>
        <body>
          <h2>Transport Status Update</h2>
          <p>Hello {{name}},</p>
          <p>There has been an update to the transport for {{deceasedName}}:</p>
          <p><strong>Status:</strong> {{status}}</p>
          <p><strong>Location:</strong> {{location}}</p>
          <p><strong>Time:</strong> {{timestamp}}</p>
          <p>You can track the transport in real-time through your Obitix family portal.</p>
          <p>Best regards,<br>The Obitix Team</p>
        </body>
        </html>
      `,
      'emergency-alert': `
        <!DOCTYPE html>
        <html>
        <head>
          <meta charset="utf-8">
          <title>Emergency Alert - Obitix</title>
        </head>
        <body>
          <h2 style="color: #dc2626;">Emergency Alert</h2>
          <p>Hello {{name}},</p>
          <p>An emergency alert has been triggered for transport {{caseId}}:</p>
          <p><strong>Alert Type:</strong> {{alertType}}</p>
          <p><strong>Description:</strong> {{description}}</p>
          <p><strong>Time:</strong> {{timestamp}}</p>
          <p>Our team is responding to this situation. You will receive updates as they become available.</p>
          <p>Best regards,<br>The Obitix Team</p>
        </body>
        </html>
      `,
      'welcome-family': `
        <!DOCTYPE html>
        <html>
        <head>
          <meta charset="utf-8">
          <title>Welcome to Obitix - Family Portal</title>
        </head>
        <body>
          <h2>Welcome to Obitix Family Portal</h2>
          <p>Hello {{name}},</p>
          <p>Welcome to the Obitix family portal. You now have access to real-time tracking and updates for your loved one's transport.</p>
          <p>Your login credentials:</p>
          <p><strong>Email:</strong> {{email}}</p>
          <p>You can access your portal at: <a href="{{portalUrl}}">{{portalUrl}}</a></p>
          <p>If you have any questions, please don't hesitate to contact us.</p>
          <p>Best regards,<br>The Obitix Team</p>
        </body>
        </html>
      `
    };
  }

  async sendEmail(options) {
    try {
      if (!this.isInitialized) {
        console.log('Email service not initialized, skipping email send');
        return {
          success: false,
          message: 'Email service not available'
        };
      }

      const { to, subject, template, data, html, text } = options;

      let emailHtml = html;
      let emailText = text;

      // Use template if provided
      if (template && this.templates[template]) {
        emailHtml = this.templates[template];
        
        // Replace template variables
        if (data) {
          Object.keys(data).forEach(key => {
            const regex = new RegExp(`{{${key}}}`, 'g');
            emailHtml = emailHtml.replace(regex, data[key]);
          });
        }
      }

      const mailOptions = {
        from: `"Obitix" <${process.env.SMTP_USER}>`,
        to,
        subject,
        html: emailHtml,
        text: emailText || this.htmlToText(emailHtml)
      };

      const result = await this.transporter.sendMail(mailOptions);
      console.log(`Email sent successfully to ${to}: ${result.messageId}`);
      
      return {
        success: true,
        messageId: result.messageId
      };
    } catch (error) {
      console.error('Failed to send email:', error);
      throw error;
    }
  }

  htmlToText(html) {
    if (!html) return '';
    return html
      .replace(/<[^>]*>/g, '')
      .replace(/&nbsp;/g, ' ')
      .replace(/&amp;/g, '&')
      .replace(/&lt;/g, '<')
      .replace(/&gt;/g, '>')
      .replace(/&quot;/g, '"')
      .trim();
  }

  // Specific email functions
  async sendVerificationEmail(email, name, verificationUrl) {
    return this.sendEmail({
      to: email,
      subject: 'Welcome to Obitix - Verify Your Email',
      template: 'email-verification',
      data: {
        name,
        verificationUrl
      }
    });
  }

  async sendPasswordResetEmail(email, name, resetUrl) {
    return this.sendEmail({
      to: email,
      subject: 'Obitix - Password Reset Request',
      template: 'password-reset',
      data: {
        name,
        resetUrl
      }
    });
  }

  async sendTransportUpdateEmail(email, name, transportData) {
    return this.sendEmail({
      to: email,
      subject: 'Transport Update - Obitix',
      template: 'transport-update',
      data: {
        name,
        deceasedName: transportData.deceasedName,
        status: transportData.status,
        location: transportData.location,
        timestamp: transportData.timestamp
      }
    });
  }

  async sendEmergencyAlertEmail(email, name, alertData) {
    return this.sendEmail({
      to: email,
      subject: 'Emergency Alert - Obitix',
      template: 'emergency-alert',
      data: {
        name,
        caseId: alertData.caseId,
        alertType: alertData.type,
        description: alertData.description,
        timestamp: alertData.timestamp
      }
    });
  }

  async sendWelcomeFamilyEmail(email, name, portalUrl) {
    return this.sendEmail({
      to: email,
      subject: 'Welcome to Obitix Family Portal',
      template: 'welcome-family',
      data: {
        name,
        email,
        portalUrl
      }
    });
  }

  async sendCustomEmail(to, subject, html, text) {
    return this.sendEmail({
      to,
      subject,
      html,
      text
    });
  }

  // Bulk email sending
  async sendBulkEmail(recipients, options) {
    const results = [];
    
    for (const recipient of recipients) {
      try {
        const result = await this.sendEmail({
          ...options,
          to: recipient.email
        });
        results.push({ email: recipient.email, success: true, result });
      } catch (error) {
        results.push({ email: recipient.email, success: false, error: error.message });
      }
    }
    
    return results;
  }

  // Email validation
  validateEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
  }

  // Get email statistics
  async getEmailStats() {
    // This would typically integrate with your email service provider's API
    // For now, return basic stats
    return {
      sentToday: 0,
      sentThisWeek: 0,
      sentThisMonth: 0,
      deliveryRate: 0.98,
      bounceRate: 0.02
    };
  }
}

// Create singleton instance
const emailService = new EmailService();

// Initialize function
const initializeEmail = async () => {
  try {
    // Check if email configuration exists
    if (!process.env.SMTP_HOST || !process.env.SMTP_USER || !process.env.SMTP_PASS) {
      console.log('Email configuration not found, email service will be disabled');
      return emailService;
    }
    
    await emailService.initialize();
    return emailService;
  } catch (error) {
    console.error('Failed to initialize email service:', error);
    console.log('Email service will be disabled');
    return emailService;
  }
};

// Export functions for use in routes
const sendEmail = (options) => emailService.sendEmail(options);
const sendVerificationEmail = (email, name, verificationUrl) => 
  emailService.sendVerificationEmail(email, name, verificationUrl);
const sendPasswordResetEmail = (email, name, resetUrl) => 
  emailService.sendPasswordResetEmail(email, name, resetUrl);
const sendTransportUpdateEmail = (email, name, transportData) => 
  emailService.sendTransportUpdateEmail(email, name, transportData);
const sendEmergencyAlertEmail = (email, name, alertData) => 
  emailService.sendEmergencyAlertEmail(email, name, alertData);
const sendWelcomeFamilyEmail = (email, name, portalUrl) => 
  emailService.sendWelcomeFamilyEmail(email, name, portalUrl);

module.exports = {
  emailService,
  initialize: initializeEmail,
  sendEmail,
  sendVerificationEmail,
  sendPasswordResetEmail,
  sendTransportUpdateEmail,
  sendEmergencyAlertEmail,
  sendWelcomeFamilyEmail,
  isInitialized: () => emailService.isInitialized
};
