const { ethers } = require('ethers');
const { Web3 } = require('web3');
const fs = require('fs');
const path = require('path');

class BlockchainService {
  constructor() {
    this.provider = null;
    this.signer = null;
    this.contract = null;
    this.web3 = null;
    this.isInitialized = false;
  }

  async initialize() {
    try {
      // Check if blockchain configuration is available
      if (!process.env.BSC_TESTNET_RPC || !process.env.PRIVATE_KEY || !process.env.CONTRACT_ADDRESS) {
        console.log('Blockchain configuration not available, skipping blockchain initialization');
        this.isInitialized = false;
        return false;
      }

      // Validate private key format
      const privateKey = process.env.PRIVATE_KEY;
      if (privateKey === 'your-private-key' || privateKey.length < 64) {
        console.log('Invalid private key format, skipping blockchain initialization');
        this.isInitialized = false;
        return false;
      }

      // Initialize Web3 provider for BSC Testnet
      this.web3 = new Web3(process.env.BSC_TESTNET_RPC);
      
      // Initialize ethers provider
      this.provider = new ethers.JsonRpcProvider(process.env.BSC_TESTNET_RPC);
      
      // Initialize signer with private key (remove 0x prefix if present)
      const cleanPrivateKey = privateKey.startsWith('0x') ? privateKey.slice(2) : privateKey;
      this.signer = new ethers.Wallet(cleanPrivateKey, this.provider);
      
      // Load smart contract ABI and address
      const contractAddress = process.env.CONTRACT_ADDRESS;
      const contractABI = this.loadContractABI();
      
      // Initialize contract instance
      this.contract = new ethers.Contract(contractAddress, contractABI, this.signer);
      
      // Test connection
      const network = await this.provider.getNetwork();
      console.log(`Connected to BSC Testnet (Chain ID: ${network.chainId})`);
      
      this.isInitialized = true;
      return true;
    } catch (error) {
      console.error('Failed to initialize blockchain service:', error);
      this.isInitialized = false;
      return false;
    }
  }

  loadContractABI() {
    try {
      // Load the compiled contract ABI
      const abiPath = path.join(__dirname, '../contracts/ObitixCustody.json');
      const contractData = JSON.parse(fs.readFileSync(abiPath, 'utf8'));
      return contractData.abi;
    } catch (error) {
      console.error('Failed to load contract ABI:', error);
      // Return a basic ABI for testing
      return [
        {
          "inputs": [
            {
              "internalType": "string",
              "name": "caseId",
              "type": "string"
            },
            {
              "internalType": "string",
              "name": "eventType",
              "type": "string"
            },
            {
              "internalType": "string",
              "name": "metadata",
              "type": "string"
            }
          ],
          "name": "recordCustodyEvent",
          "outputs": [
            {
              "internalType": "bytes32",
              "name": "",
              "type": "bytes32"
            }
          ],
          "stateMutability": "nonpayable",
          "type": "function"
        },
        {
          "inputs": [
            {
              "internalType": "string",
              "name": "caseId",
              "type": "string"
            }
          ],
          "name": "getCustodyEvents",
          "outputs": [
            {
              "components": [
                {
                  "internalType": "string",
                  "name": "eventType",
                  "type": "string"
                },
                {
                  "internalType": "string",
                  "name": "metadata",
                  "type": "string"
                },
                {
                  "internalType": "uint256",
                  "name": "timestamp",
                  "type": "uint256"
                },
                {
                  "internalType": "address",
                  "name": "recordedBy",
                  "type": "address"
                }
              ],
              "internalType": "struct ObitixCustody.CustodyEvent[]",
              "name": "",
              "type": "tuple[]"
            }
          ],
          "stateMutability": "view",
          "type": "function"
        }
      ];
    }
  }

  async recordCustodyEvent(caseId, eventType, metadata) {
    try {
      if (!this.isInitialized) {
        throw new Error('Blockchain service not initialized');
      }

      console.log(`Recording custody event: ${eventType} for case: ${caseId}`);

      // Prepare transaction
      const tx = await this.contract.recordCustodyEvent(
        caseId,
        eventType,
        metadata
      );

      // Wait for transaction confirmation
      const receipt = await tx.wait();
      
      console.log(`Custody event recorded. Transaction hash: ${receipt.hash}`);

      return {
        success: true,
        transactionHash: receipt.hash,
        blockNumber: receipt.blockNumber,
        eventId: receipt.logs[0]?.topics[1] || null
      };
    } catch (error) {
      console.error('Failed to record custody event:', error);
      throw error;
    }
  }

  async getCustodyEvents(caseId) {
    try {
      if (!this.isInitialized) {
        throw new Error('Blockchain service not initialized');
      }

      const events = await this.contract.getCustodyEvents(caseId);
      
      return events.map(event => ({
        eventType: event.eventType,
        metadata: event.metadata,
        timestamp: new Date(parseInt(event.timestamp) * 1000),
        recordedBy: event.recordedBy
      }));
    } catch (error) {
      console.error('Failed to get custody events:', error);
      throw error;
    }
  }

  async verifyCustodyChain(caseId) {
    try {
      if (!this.isInitialized) {
        throw new Error('Blockchain service not initialized');
      }

      const events = await this.getCustodyEvents(caseId);
      
      // Verify chain integrity
      let isValid = true;
      let lastTimestamp = 0;
      
      for (const event of events) {
        if (event.timestamp.getTime() < lastTimestamp) {
          isValid = false;
          break;
        }
        lastTimestamp = event.timestamp.getTime();
      }

      return {
        isValid,
        eventCount: events.length,
        events: events
      };
    } catch (error) {
      console.error('Failed to verify custody chain:', error);
      throw error;
    }
  }

  async getContractBalance() {
    try {
      if (!this.isInitialized) {
        throw new Error('Blockchain service not initialized');
      }

      const balance = await this.provider.getBalance(this.contract.target);
      return ethers.formatEther(balance);
    } catch (error) {
      console.error('Failed to get contract balance:', error);
      throw error;
    }
  }

  async getAccountBalance(address) {
    try {
      if (!this.isInitialized) {
        throw new Error('Blockchain service not initialized');
      }

      const balance = await this.provider.getBalance(address);
      return ethers.formatEther(balance);
    } catch (error) {
      console.error('Failed to get account balance:', error);
      throw error;
    }
  }

  async estimateGas(caseId, eventType, metadata) {
    try {
      if (!this.isInitialized) {
        throw new Error('Blockchain service not initialized');
      }

      const gasEstimate = await this.contract.recordCustodyEvent.estimateGas(
        caseId,
        eventType,
        metadata
      );

      return gasEstimate.toString();
    } catch (error) {
      console.error('Failed to estimate gas:', error);
      throw error;
    }
  }

  async getTransactionReceipt(txHash) {
    try {
      if (!this.isInitialized) {
        throw new Error('Blockchain service not initialized');
      }

      const receipt = await this.provider.getTransactionReceipt(txHash);
      return receipt;
    } catch (error) {
      console.error('Failed to get transaction receipt:', error);
      throw error;
    }
  }

  // Utility method to create a new wallet
  createWallet() {
    const wallet = ethers.Wallet.createRandom();
    return {
      address: wallet.address,
      privateKey: wallet.privateKey,
      mnemonic: wallet.mnemonic?.phrase
    };
  }

  // Utility method to validate address
  isValidAddress(address) {
    return ethers.isAddress(address);
  }

  // Utility method to format address
  formatAddress(address) {
    return ethers.getAddress(address);
  }

  // Method to handle contract events
  async listenToEvents(eventName, callback) {
    try {
      if (!this.isInitialized) {
        throw new Error('Blockchain service not initialized');
      }

      this.contract.on(eventName, (...args) => {
        callback(args);
      });

      console.log(`Listening to ${eventName} events`);
    } catch (error) {
      console.error(`Failed to listen to ${eventName} events:`, error);
      throw error;
    }
  }

  // Method to stop listening to events
  async stopListeningToEvents(eventName) {
    try {
      if (!this.isInitialized) {
        throw new Error('Blockchain service not initialized');
      }

      this.contract.off(eventName);
      console.log(`Stopped listening to ${eventName} events`);
    } catch (error) {
      console.error(`Failed to stop listening to ${eventName} events:`, error);
      throw error;
    }
  }

  // Method to get network information
  async getNetworkInfo() {
    try {
      if (!this.isInitialized) {
        throw new Error('Blockchain service not initialized');
      }

      const network = await this.provider.getNetwork();
      const blockNumber = await this.provider.getBlockNumber();
      const gasPrice = await this.provider.getFeeData();

      return {
        chainId: network.chainId,
        name: network.name,
        blockNumber: blockNumber,
        gasPrice: gasPrice.gasPrice ? ethers.formatUnits(gasPrice.gasPrice, 'gwei') : null
      };
    } catch (error) {
      console.error('Failed to get network info:', error);
      throw error;
    }
  }
}

// Create singleton instance
const blockchainService = new BlockchainService();

// Initialize function
const initializeBlockchain = async () => {
  try {
    await blockchainService.initialize();
    console.log('Blockchain service initialized successfully');
    return blockchainService;
  } catch (error) {
    console.error('Failed to initialize blockchain service:', error);
    throw error;
  }
};

module.exports = {
  blockchainService,
  initializeBlockchain
};
