const OpenAI = require('openai');
const axios = require('axios');

class AIService {
  constructor() {
    this.openai = null;
    this.retellApiKey = process.env.RETELL_API_KEY;
    this.retellAgentId = process.env.RETELL_AGENT_ID;
    this.isInitialized = false;
  }

  async initialize() {
    try {
      // Initialize OpenAI
      if (!process.env.OPENAI_API_KEY) {
        console.log('OpenAI API key not configured, skipping AI initialization');
        this.isInitialized = false;
        return false;
      }

      this.openai = new OpenAI({
        apiKey: process.env.OPENAI_API_KEY,
      });

      // Test OpenAI connection
      await this.openai.models.list();
      console.log('OpenAI service initialized successfully');

      // Test Retell.ai connection (optional)
      if (this.retellApiKey) {
        try {
          await this.testRetellConnection();
          console.log('Retell.ai service initialized successfully');
        } catch (retellError) {
          console.warn('Retell.ai connection failed, continuing without Retell support:', retellError.message);
        }
      } else {
        console.log('Retell.ai API key not configured, skipping Retell initialization');
      }

      this.isInitialized = true;
      return true;
    } catch (error) {
      console.error('Failed to initialize AI service:', error);
      this.isInitialized = false;
      return false;
    }
  }

  async testRetellConnection() {
    try {
      // Test actual Retell.ai connection - use the correct endpoint
      const response = await axios.get('https://api.retellai.com/v1/agents', {
        headers: {
          'Authorization': `Bearer ${this.retellApiKey}`,
          'Content-Type': 'application/json'
        }
      });
      console.log('Retell.ai connection test successful');
      return response.data;
    } catch (error) {
      console.error('Retell.ai connection test failed:', error);
      // Don't throw error, just log it and continue
      console.log('Retell.ai connection failed, continuing without Retell support:', error.message);
      return { success: false, message: 'Retell.ai not available' };
    }
  }

  // OpenAI Chat Completion for general queries
  async chatCompletion(messages, options = {}) {
    try {
      if (!this.isInitialized) {
        throw new Error('AI service not initialized');
      }

      const defaultOptions = {
        model: 'gpt-4o-mini', // Use available model instead of gpt-4
        max_tokens: 1000,
        temperature: 0.7,
        ...options
      };

      const response = await this.openai.chat.completions.create({
        messages,
        ...defaultOptions
      });

      return {
        success: true,
        content: response.choices[0].message.content,
        usage: response.usage,
        model: response.model
      };
    } catch (error) {
      console.error('OpenAI chat completion failed:', error);
      throw error;
    }
  }

  // Family member support chat
  async familySupportChat(userMessage, transportContext = null) {
    try {
      const systemPrompt = `You are an empathetic AI assistant for Obitix, a funeral home chain-of-custody tracking platform. 
      You help family members with questions about their loved one's transport status and provide compassionate support.
      
      Guidelines:
      - Be empathetic and understanding
      - Provide clear, accurate information about transport status
      - Offer reassurance and support
      - Direct complex questions to human staff when appropriate
      - Maintain confidentiality and respect
      
      Current transport context: ${transportContext ? JSON.stringify(transportContext) : 'None provided'}`;

      const messages = [
        { role: 'system', content: systemPrompt },
        { role: 'user', content: userMessage }
      ];

      return await this.chatCompletion(messages, {
        temperature: 0.8,
        max_tokens: 800
      });
    } catch (error) {
      console.error('Family support chat failed:', error);
      throw error;
    }
  }

  // Technician voice logging and task assistance
  async technicianVoiceAssistant(audioInput, context = {}) {
    try {
      if (!this.isInitialized) {
        throw new Error('AI service not initialized');
      }

      // First, transcribe the audio
      const transcription = await this.transcribeAudio(audioInput);
      
      // Then process the transcription for task extraction
      const taskAnalysis = await this.analyzeTechnicianTask(transcription, context);
      
      return {
        success: true,
        transcription: transcription.content,
        taskAnalysis,
        confidence: transcription.confidence
      };
    } catch (error) {
      console.error('Technician voice assistant failed:', error);
      throw error;
    }
  }

  // Audio transcription
  async transcribeAudio(audioBuffer, options = {}) {
    try {
      if (!this.isInitialized) {
        throw new Error('AI service not initialized');
      }

      const response = await this.openai.audio.transcriptions.create({
        file: audioBuffer,
        model: 'whisper-1',
        language: options.language || 'en',
        prompt: options.prompt || 'This is a conversation about funeral transport and chain-of-custody procedures.',
        response_format: 'verbose_json',
        temperature: 0.2
      });

      return {
        success: true,
        content: response.text,
        confidence: response.avg_logprob || 0,
        language: response.language,
        duration: response.duration
      };
    } catch (error) {
      console.error('Audio transcription failed:', error);
      throw error;
    }
  }

  // Analyze technician tasks from voice input
  async analyzeTechnicianTask(transcription, context = {}) {
    try {
      const systemPrompt = `You are an AI assistant that analyzes technician voice logs for funeral transport tasks.
      Extract key information and identify required actions from the technician's voice input.
      
      Context: ${JSON.stringify(context)}
      
      Analyze the following voice input and extract:
      1. Task type (pickup, delivery, checkpoint, emergency, etc.)
      2. Location information
      3. Time stamps
      4. Required actions
      5. Any issues or delays
      6. Compliance requirements
      
      Return a structured JSON response with these fields.`;

      const messages = [
        { role: 'system', content: systemPrompt },
        { role: 'user', content: transcription }
      ];

      const response = await this.chatCompletion(messages, {
        temperature: 0.3,
        max_tokens: 500
      });

      // Parse the JSON response
      try {
        const taskAnalysis = JSON.parse(response.content);
        return {
          ...taskAnalysis,
          rawTranscription: transcription
        };
      } catch (parseError) {
        // If JSON parsing fails, return structured response
        return {
          taskType: 'unknown',
          location: null,
          timestamp: new Date().toISOString(),
          actions: [],
          issues: [],
          compliance: [],
          rawTranscription: transcription,
          parseError: true
        };
      }
    } catch (error) {
      console.error('Task analysis failed:', error);
      throw error;
    }
  }

  // First call intake automation
  async firstCallIntake(callTranscript, options = {}) {
    try {
      const systemPrompt = `You are an AI assistant for funeral home first call intake.
      Extract key information from the call transcript and create a structured intake record.
      
      Extract the following information:
      1. Deceased person details (name, age, date of death, place of death)
      2. Contact person information
      3. Location details (pickup and delivery)
      4. Special requirements or conditions
      5. Urgency level
      6. Next steps required
      
      Return a structured JSON response.`;

      const messages = [
        { role: 'system', content: systemPrompt },
        { role: 'user', content: callTranscript }
      ];

      const response = await this.chatCompletion(messages, {
        temperature: 0.2,
        max_tokens: 800
      });

      try {
        const intakeData = JSON.parse(response.content);
        return {
          success: true,
          intakeData,
          confidence: response.usage ? response.usage.total_tokens / 1000 : 0
        };
      } catch (parseError) {
        return {
          success: false,
          error: 'Failed to parse intake data',
          rawResponse: response.content
        };
      }
    } catch (error) {
      console.error('First call intake failed:', error);
      throw error;
    }
  }

  // FAQ automation
  async faqAssistant(question, context = {}) {
    try {
      const systemPrompt = `You are an AI assistant for Obitix funeral home services.
      Answer questions about funeral transport, chain-of-custody, and related services.
      
      Common topics:
      - Transport procedures and timing
      - Required documents and permits
      - Family tracking and notifications
      - Compliance and regulations
      - Emergency procedures
      
      Context: ${JSON.stringify(context)}
      
      Provide clear, accurate, and empathetic responses.`;

      const messages = [
        { role: 'system', content: systemPrompt },
        { role: 'user', content: question }
      ];

      return await this.chatCompletion(messages, {
        temperature: 0.7,
        max_tokens: 600
      });
    } catch (error) {
      console.error('FAQ assistant failed:', error);
      throw error;
    }
  }

  // Retell.ai voice agent integration
  async createRetellCall(phoneNumber, agentId = null, message = null, callType = 'notification') {
    try {
      if (!this.retellApiKey) {
        throw new Error('Retell.ai API key not configured');
      }

      // Prepare call metadata with message content
      const metadata = {
        platform: 'obitix',
        timestamp: new Date().toISOString(),
        callType: callType
      };

      // If message is provided, include it in metadata for the AI agent
      if (message) {
        metadata.message = message;
        metadata.context = `This is a ${callType} call from Obitix. The message to deliver is: ${message}`;
      }

      const response = await axios.post('https://api.retellai.com/call', {
        agent_id: agentId || this.retellAgentId,
        phone_number: phoneNumber,
        metadata: metadata
      }, {
        headers: {
          'Authorization': `Bearer ${this.retellApiKey}`,
          'Content-Type': 'application/json'
        }
      });

      return {
        success: true,
        callId: response.data.call_id,
        status: response.data.status,
        callType: callType
      };
    } catch (error) {
      console.error('Retell.ai call creation failed:', error);
      throw error;
    }
  }

  // Create intelligent voice notification using Retell
  async createIntelligentVoiceNotification(phoneNumber, message, context = {}) {
    try {
      if (!this.retellApiKey) {
        throw new Error('Retell.ai API key not configured');
      }

      // Enhance the message with context using OpenAI
      const enhancedMessage = await this.enhanceNotificationMessage(message, context);
      
      // Create Retell call with enhanced message
      return await this.createRetellCall(phoneNumber, null, enhancedMessage, 'notification');
    } catch (error) {
      console.error('Intelligent voice notification failed:', error);
      throw error;
    }
  }

  // Enhance notification message with context using OpenAI
  async enhanceNotificationMessage(message, context = {}) {
    try {
      const systemPrompt = `You are an AI assistant that enhances notification messages for a funeral transport platform called Obitix.
      Your task is to make the message more natural, compassionate, and contextually appropriate.
      
      Context: ${JSON.stringify(context)}
      Original message: ${message}
      
      Enhance the message to be:
      1. More natural and conversational
      2. Compassionate and respectful
      3. Clear and informative
      4. Appropriate for voice delivery
      5. Under 30 seconds when spoken
      
      Return only the enhanced message, no additional formatting.`;

      const response = await this.chatCompletion([
        { role: 'system', content: systemPrompt },
        { role: 'user', content: 'Please enhance this notification message.' }
      ], {
        temperature: 0.7,
        max_tokens: 200
      });

      return response.content.trim();
    } catch (error) {
      console.error('Message enhancement failed:', error);
      // Return original message if enhancement fails
      return message;
    }
  }

  // Get Retell.ai call status
  async getRetellCallStatus(callId) {
    try {
      if (!this.retellApiKey) {
        throw new Error('Retell.ai API key not configured');
      }

      const response = await axios.get(`https://api.retellai.com/call/${callId}`, {
        headers: {
          'Authorization': `Bearer ${this.retellApiKey}`,
          'Content-Type': 'application/json'
        }
      });

      return {
        success: true,
        callData: response.data
      };
    } catch (error) {
      console.error('Retell.ai call status check failed:', error);
      throw error;
    }
  }

  // Generate task reminders
  async generateTaskReminders(transportData) {
    try {
      const systemPrompt = `You are an AI assistant that generates task reminders for funeral transport technicians.
      Based on the transport data, generate a list of tasks that need to be completed and their estimated timing.
      
      Transport data: ${JSON.stringify(transportData)}
      
      Generate reminders for:
      1. Pre-pickup tasks
      2. Pickup procedures
      3. Transport checkpoints
      4. Delivery procedures
      5. Post-delivery tasks
      
      Return a structured JSON with tasks and estimated times.`;

      const messages = [
        { role: 'system', content: systemPrompt },
        { role: 'user', content: 'Generate task reminders for this transport.' }
      ];

      const response = await this.chatCompletion(messages, {
        temperature: 0.3,
        max_tokens: 600
      });

      try {
        const reminders = JSON.parse(response.content);
        return {
          success: true,
          reminders,
          transportId: transportData.caseId
        };
      } catch (parseError) {
        return {
          success: false,
          error: 'Failed to parse reminders',
          rawResponse: response.content
        };
      }
    } catch (error) {
      console.error('Task reminder generation failed:', error);
      throw error;
    }
  }

  // Compliance alert generation
  async generateComplianceAlerts(transportData) {
    try {
      const systemPrompt = `You are an AI assistant that monitors funeral transport compliance.
      Analyze the transport data and identify any potential compliance issues or required actions.
      
      Transport data: ${JSON.stringify(transportData)}
      
      Check for:
      1. Missing required documents
      2. Timeline compliance issues
      3. Regulatory requirements
      4. Safety concerns
      5. Required notifications
      
      Return a structured JSON with alerts and recommendations.`;

      const messages = [
        { role: 'system', content: systemPrompt },
        { role: 'user', content: 'Analyze compliance for this transport.' }
      ];

      const response = await this.chatCompletion(messages, {
        temperature: 0.2,
        max_tokens: 500
      });

      try {
        const alerts = JSON.parse(response.content);
        return {
          success: true,
          alerts,
          transportId: transportData.caseId
        };
      } catch (parseError) {
        return {
          success: false,
          error: 'Failed to parse compliance alerts',
          rawResponse: response.content
        };
      }
    } catch (error) {
      console.error('Compliance alert generation failed:', error);
      throw error;
    }
  }

  // Voice synthesis for notifications
  async synthesizeVoice(text, options = {}) {
    try {
      if (!this.isInitialized) {
        throw new Error('AI service not initialized');
      }

      const response = await this.openai.audio.speech.create({
        model: 'tts-1',
        voice: options.voice || 'alloy',
        input: text,
        speed: options.speed || 1.0
      });

      return {
        success: true,
        audioBuffer: response,
        format: 'mp3',
        duration: options.duration || null
      };
    } catch (error) {
      console.error('Voice synthesis failed:', error);
      throw error;
    }
  }

  // Sentiment analysis for family communications
  async analyzeSentiment(text) {
    try {
      const systemPrompt = `Analyze the sentiment of the following text from a family member.
      Classify as: positive, negative, neutral, anxious, grateful, or confused.
      Provide a confidence score (0-1) and brief explanation.`;

      const messages = [
        { role: 'system', content: systemPrompt },
        { role: 'user', content: text }
      ];

      const response = await this.chatCompletion(messages, {
        temperature: 0.3,
        max_tokens: 200
      });

      try {
        const sentiment = JSON.parse(response.content);
        return {
          success: true,
          sentiment
        };
      } catch (parseError) {
        return {
          success: false,
          error: 'Failed to parse sentiment analysis',
          rawResponse: response.content
        };
      }
    } catch (error) {
      console.error('Sentiment analysis failed:', error);
      throw error;
    }
  }
}

// Create singleton instance
const aiService = new AIService();

// Initialize function
const initializeAI = async () => {
  try {
    await aiService.initialize();
    console.log('AI service initialized successfully');
    return aiService;
  } catch (error) {
    console.error('Failed to initialize AI service:', error);
    throw error;
  }
};

module.exports = {
  aiService,
  initializeAI
};
