const express = require('express');
const router = express.Router();
const stripe = require('stripe')(process.env.STRIPE_SECRET_KEY);
const { verifyToken, authorize } = require('../middleware/auth');
const stripeService = require('../services/stripe');
const Subscription = require('../models/Subscription');
const Organization = require('../models/Organization');

// Get subscription pricing
router.get('/pricing', async (req, res) => {
  try {
    const pricing = stripeService.getPricing();
    res.json({
      success: true,
      data: pricing
    });
  } catch (error) {
    console.error('Error getting pricing:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to get pricing information'
    });
  }
});



// Get organization subscription
router.get('/organization', verifyToken, authorize(['admin', 'funeral_home_staff']), async (req, res) => {
  try {
    const subscription = await stripeService.getOrganizationSubscription(req.user.organizationId);
    
    if (!subscription) {
      return res.json({
        success: true,
        data: null,
        message: 'No active subscription found for this organization'
      });
    }

    res.json({
      success: true,
      data: subscription
    });
  } catch (error) {
    console.error('Error getting organization subscription:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to get subscription information'
    });
  }
});

// Create checkout session
router.post('/create-checkout-session', verifyToken, authorize(['admin']), async (req, res) => {
  try {
    const { tier, successUrl, cancelUrl } = req.body;

    if (!tier || !['basic', 'professional', 'enterprise'].includes(tier)) {
      return res.status(400).json({
        success: false,
        message: 'Invalid tier specified'
      });
    }

    // Check if Stripe is properly configured
    if (!process.env.STRIPE_SECRET_KEY || process.env.STRIPE_SECRET_KEY.includes('your-')) {
      return res.status(500).json({
        success: false,
        message: 'Stripe is not properly configured. Please contact support.'
      });
    }

    // Check if price IDs are configured
    const priceId = tier === 'basic' ? process.env.STRIPE_BASIC_PRICE_ID :
                   tier === 'professional' ? process.env.STRIPE_PROFESSIONAL_PRICE_ID :
                   process.env.STRIPE_ENTERPRISE_PRICE_ID;

    if (!priceId || priceId.includes('your_')) {
      return res.status(500).json({
        success: false,
        message: `Stripe price ID for ${tier} tier is not configured. Please contact support.`
      });
    }

    // Set default URLs if not provided
    const baseUrl = 'https://obitix.prohirely.ltd';
    const defaultSuccessUrl = successUrl ? 
      (successUrl.startsWith('http') ? successUrl : `${baseUrl}${successUrl}`) : 
      `${baseUrl}/admin/subscription/success`;
    const defaultCancelUrl = cancelUrl ? 
      (cancelUrl.startsWith('http') ? cancelUrl : `${baseUrl}${cancelUrl}`) : 
      `${baseUrl}/admin/subscription/cancel`;

    const session = await stripeService.createCheckoutSession(
      req.user.organizationId,
      tier,
      defaultSuccessUrl,
      defaultCancelUrl
    );

    res.json({
      success: true,
      data: {
        sessionId: session.id,
        url: session.url
      }
    });
  } catch (error) {
    console.error('Error creating checkout session:', error);
    
    // Provide more specific error messages
    let errorMessage = 'Failed to create checkout session';
    if (error.message.includes('Invalid API Key')) {
      errorMessage = 'Stripe is not properly configured. Please contact support.';
    } else if (error.message.includes('No such price')) {
      errorMessage = 'Subscription pricing is not configured. Please contact support.';
    } else if (error.message.includes('Organization not found')) {
      errorMessage = 'Organization not found. Please contact support.';
    }

    res.status(500).json({
      success: false,
      message: errorMessage
    });
  }
});

// Update subscription tier
router.put('/:subscriptionId/update', verifyToken, authorize(['admin']), async (req, res) => {
  try {
    const { subscriptionId } = req.params;
    const { tier } = req.body;

    if (!tier || !['basic', 'professional', 'enterprise'].includes(tier)) {
      return res.status(400).json({
        success: false,
        message: 'Invalid tier specified'
      });
    }

    const subscription = await stripeService.updateSubscription(subscriptionId, tier);

    res.json({
      success: true,
      data: subscription,
      message: 'Subscription updated successfully'
    });
  } catch (error) {
    console.error('Error updating subscription:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update subscription'
    });
  }
});

// Cancel subscription
router.post('/:subscriptionId/cancel', verifyToken, authorize(['admin']), async (req, res) => {
  try {
    const { subscriptionId } = req.params;
    const { cancelAtPeriodEnd = true } = req.body;

    const subscription = await stripeService.cancelSubscription(subscriptionId, cancelAtPeriodEnd);

    res.json({
      success: true,
      data: subscription,
      message: cancelAtPeriodEnd ? 'Subscription will be canceled at the end of the current period' : 'Subscription canceled immediately'
    });
  } catch (error) {
    console.error('Error canceling subscription:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to cancel subscription'
    });
  }
});

// Reactivate subscription
router.post('/:subscriptionId/reactivate', verifyToken, authorize(['admin']), async (req, res) => {
  try {
    const { subscriptionId } = req.params;

    const subscription = await stripeService.reactivateSubscription(subscriptionId);

    res.json({
      success: true,
      data: subscription,
      message: 'Subscription reactivated successfully'
    });
  } catch (error) {
    console.error('Error reactivating subscription:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to reactivate subscription'
    });
  }
});

// Get subscription usage
router.get('/:subscriptionId/usage', verifyToken, authorize(['admin', 'funeral_home_staff']), async (req, res) => {
  try {
    const { subscriptionId } = req.params;

    const subscription = await Subscription.findById(subscriptionId);
    if (!subscription) {
      return res.status(404).json({
        success: false,
        message: 'Subscription not found'
      });
    }

    // Get current usage from organization
    const organization = await Organization.findById(req.user.organizationId);
    
    // Calculate current usage (this would need to be implemented based on your data)
    const currentUsage = {
      transportsThisMonth: 0, // TODO: Calculate from transport data
      technicians: 0, // TODO: Calculate from user data
      locations: 0, // TODO: Calculate from location data
      apiCallsThisMonth: 0, // TODO: Track API calls
      storageUsedGB: 0 // TODO: Calculate from document storage
    };

    // Calculate usage percentages
    const usagePercentages = {};
    Object.keys(currentUsage).forEach(key => {
      const limit = subscription.limits[key];
      if (limit === -1) {
        usagePercentages[key] = 0; // Unlimited
      } else {
        usagePercentages[key] = Math.min((currentUsage[key] / limit) * 100, 100);
      }
    });

    res.json({
      success: true,
      data: {
        subscription,
        currentUsage,
        usagePercentages,
        limits: subscription.limits
      }
    });
  } catch (error) {
    console.error('Error getting subscription usage:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to get subscription usage'
    });
  }
});

// Check feature access (POST - for specific feature)
router.post('/check-feature', verifyToken, async (req, res) => {
  try {
    const { featureName } = req.body;

    if (!featureName) {
      return res.status(400).json({
        success: false,
        message: 'Feature name is required'
      });
    }

    const subscription = await Subscription.findOne({ organizationId: req.user.organizationId });
    
    if (!subscription) {
      return res.json({
        success: true,
        data: {
          hasAccess: false,
          message: 'No active subscription found'
        }
      });
    }

    const hasAccess = subscription.hasFeature(featureName);

    res.json({
      success: true,
      data: {
        hasAccess,
        subscription: {
          tier: subscription.tier,
          status: subscription.status,
          features: subscription.features
        }
      }
    });
  } catch (error) {
    console.error('Error checking feature access:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to check feature access'
    });
  }
});

// Check subscription status (GET - for general subscription check)
router.get('/check-feature', verifyToken, async (req, res) => {
  try {
    const subscription = await Subscription.findOne({ organizationId: req.user.organizationId });
    
    if (!subscription) {
      return res.json({
        success: true,
        data: {
          hasAccess: false,
          message: 'No active subscription found'
        }
      });
    }

    // Check if subscription is active
    if (subscription.status !== 'active') {
      return res.json({
        success: true,
        data: {
          hasAccess: false,
          message: `Subscription is ${subscription.status}`
        }
      });
    }

    res.json({
      success: true,
      data: {
        hasAccess: true,
        subscription: {
          tier: subscription.tier,
          status: subscription.status,
          features: subscription.features
        }
      }
    });
  } catch (error) {
    console.error('Error checking subscription status:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to check subscription status'
    });
  }
});

// Stripe webhook handler
router.post('/webhook', express.raw({ type: 'application/json' }), async (req, res) => {
  const sig = req.headers['stripe-signature'];
  const endpointSecret = process.env.STRIPE_WEBHOOK_SECRET;

  let event;

  try {
    event = stripe.webhooks.constructEvent(req.body, sig, endpointSecret);
  } catch (err) {
    console.error('Webhook signature verification failed:', err.message);
    return res.status(400).send(`Webhook Error: ${err.message}`);
  }

  try {
    await stripeService.handleWebhook(event);
    res.json({ received: true });
  } catch (error) {
    console.error('Error handling webhook:', error);
    res.status(500).json({ error: 'Webhook handling failed' });
  }
});

// Success page
router.get('/success', (req, res) => {
  res.send(`
    <!DOCTYPE html>
    <html>
    <head>
      <title>Subscription Success</title>
      <style>
        body { font-family: Arial, sans-serif; text-align: center; padding: 50px; }
        .success { color: #28a745; font-size: 24px; margin-bottom: 20px; }
        .message { color: #666; margin-bottom: 30px; }
        .button { background: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; }
        .countdown { color: #999; font-size: 14px; margin-top: 20px; }
      </style>
    </head>
    <body>
      <div class="success">✅ Payment Successful!</div>
      <div class="message">Your subscription has been activated successfully.</div>
      <div class="countdown">This window will close automatically in <span id="timer">5</span> seconds...</div>
      <a href="javascript:window.close()" class="button">Close Now</a>
      
      <script>
        let timeLeft = 5;
        const timerElement = document.getElementById('timer');
        
        const countdown = setInterval(() => {
          timeLeft--;
          timerElement.textContent = timeLeft;
          
          if (timeLeft <= 0) {
            clearInterval(countdown);
            window.close();
          }
        }, 1000);
      </script>
    </body>
    </html>
  `);
});

// Cancel page
router.get('/cancel', (req, res) => {
  res.send(`
    <!DOCTYPE html>
    <html>
    <head>
      <title>Payment Cancelled</title>
      <style>
        body { font-family: Arial, sans-serif; text-align: center; padding: 50px; }
        .cancel { color: #dc3545; font-size: 24px; margin-bottom: 20px; }
        .message { color: #666; margin-bottom: 30px; }
        .button { background: #6c757d; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; }
        .countdown { color: #999; font-size: 14px; margin-top: 20px; }
      </style>
    </head>
    <body>
      <div class="cancel">❌ Payment Cancelled</div>
      <div class="message">Your payment was cancelled. No charges were made.</div>
      <div class="countdown">This window will close automatically in <span id="timer">3</span> seconds...</div>
      <a href="javascript:window.close()" class="button">Close Now</a>
      
      <script>
        let timeLeft = 3;
        const timerElement = document.getElementById('timer');
        
        const countdown = setInterval(() => {
          timeLeft--;
          timerElement.textContent = timeLeft;
          
          if (timeLeft <= 0) {
            clearInterval(countdown);
            window.close();
          }
        }, 1000);
      </script>
    </body>
    </html>
  `);
});

module.exports = router;
