const express = require('express');
const router = express.Router();
const multer = require('multer');
const { verifyToken, authorize } = require('../middleware/auth');
const { checkFeatureAccess } = require('../middleware/subscription');
const qrCodeService = require('../services/qrCode');
const QRCodeModel = require('../models/QRCode');

// Configure multer for photo uploads
const storage = multer.memoryStorage();
const upload = multer({
  storage: storage,
  limits: {
    fileSize: 10 * 1024 * 1024, // 10MB limit for photos
  },
  fileFilter: (req, file, cb) => {
    const allowedTypes = /jpeg|jpg|png|gif/;
    const extname = allowedTypes.test(file.originalname.toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);

    if (mimetype && extname) {
      return cb(null, true);
    } else {
      cb(new Error('Only image files are allowed'));
    }
  }
});

/**
 * @route   POST /api/qr-codes/generate
 * @desc    Generate QR codes for a transport case
 * @access  Private (Admin, Staff)
 */
router.post('/generate', 
  verifyToken, 
  authorize(['admin', 'staff', 'technician', 'funeral_home_staff']), 
  checkFeatureAccess('qrCodeGeneration'),
  async (req, res) => {
    try {
      const { transportId } = req.body;
      const { id: generatedBy, organizationId } = req.user;

      if (!transportId) {
        return res.status(400).json({
          success: false,
          message: 'Transport ID is required'
        });
      }

      const qrCodes = await qrCodeService.generateTransportQRCodes(
        transportId, 
        organizationId, 
        generatedBy
      );

      res.json({
        success: true,
        message: 'QR codes generated successfully',
        data: {
          qrCodes,
          count: qrCodes.length
        }
      });
    } catch (error) {
      console.error('Error generating QR codes:', error);
      res.status(500).json({
        success: false,
        message: 'Failed to generate QR codes',
        error: error.message
      });
    }
  }
);

/**
 * @route   POST /api/qr-codes/scan
 * @desc    Scan QR code and record the event
 * @access  Private (Technician, Admin, Staff)
 */
router.post('/scan',
  verifyToken,
  authorize(['technician', 'admin', 'staff', 'funeral_home_staff']),
  checkFeatureAccess('qrCodeGeneration'),
  upload.array('photos', 5), // Allow up to 5 photos
  async (req, res) => {
    try {
      const { qrCodeId, notes, voiceLogs } = req.body;
      const { id: scannedBy } = req.user;

      if (!qrCodeId) {
        return res.status(400).json({
          success: false,
          message: 'QR code ID is required'
        });
      }

      // Get current location (from request or GPS)
      const location = {
        coordinates: req.body.coordinates ? JSON.parse(req.body.coordinates) : [0, 0],
        address: req.body.address || 'Unknown location'
      };

      // Process uploaded photos
      const photos = [];
      if (req.files && req.files.length > 0) {
        for (const file of req.files) {
          photos.push({
            data: file.buffer.toString('base64'),
            gps: {
              latitude: parseFloat(req.body.latitude) || 0,
              longitude: parseFloat(req.body.longitude) || 0,
              accuracy: parseFloat(req.body.accuracy) || 0
            },
            device: req.body.device || 'Unknown',
            orientation: req.body.orientation || 'portrait'
          });
        }
      }

      // Process voice logs
      const processedVoiceLogs = [];
      if (voiceLogs) {
        try {
          const voiceLogsArray = JSON.parse(voiceLogs);
          processedVoiceLogs.push(...voiceLogsArray);
        } catch (e) {
          console.error('Error parsing voice logs:', e);
        }
      }

      const scanData = {
        scannedBy,
        location,
        photos,
        voiceLogs: processedVoiceLogs,
        notes
      };

      const result = await qrCodeService.scanQRCode(qrCodeId, scanData);

      res.json({
        success: true,
        message: 'QR code scanned successfully',
        data: result
      });
    } catch (error) {
      console.error('Error scanning QR code:', error);
      res.status(500).json({
        success: false,
        message: 'Failed to scan QR code',
        error: error.message
      });
    }
  }
);

/**
 * @route   GET /api/qr-codes/transport/:transportId
 * @desc    Get QR codes for a specific transport
 * @access  Private (Admin, Staff, Technician)
 */
router.get('/transport/:transportId',
  verifyToken,
  authorize(['admin', 'staff', 'technician', 'funeral_home_staff']),
  checkFeatureAccess('qrCodeGeneration'),
  async (req, res) => {
    try {
      const { transportId } = req.params;
      const { organizationId } = req.user;

      // Verify transport belongs to organization
      const Transport = require('../models/Transport');
      const transport = await Transport.findOne({ 
        _id: transportId, 
        organizationId 
      });

      if (!transport) {
        return res.status(404).json({
          success: false,
          message: 'Transport not found'
        });
      }

      const qrCodes = await qrCodeService.getTransportQRCodes(transportId);

      res.json({
        success: true,
        data: {
          qrCodes,
          count: qrCodes.length,
          transport: {
            caseId: transport.caseId,
            status: transport.status,
            deceasedName: transport.deceasedName
          }
        }
      });
    } catch (error) {
      console.error('Error getting transport QR codes:', error);
      res.status(500).json({
        success: false,
        message: 'Failed to get transport QR codes',
        error: error.message
      });
    }
  }
);

/**
 * @route   GET /api/qr-codes/verify/:qrCodeId
 * @desc    Verify QR code validity
 * @access  Private (Admin, Staff, Technician)
 */
router.get('/verify/:qrCodeId',
  verifyToken,
  authorize(['admin', 'staff', 'technician', 'funeral_home_staff']),
  checkFeatureAccess('qrCodeGeneration'),
  async (req, res) => {
    try {
      const { qrCodeId } = req.params;
      const { organizationId } = req.user;

      // Find QR code and verify it belongs to organization
      const qrCode = await QRCodeModel.findOne({ 
        qrCodeId, 
        organizationId 
      });

      if (!qrCode) {
        return res.status(404).json({
          success: false,
          message: 'QR code not found'
        });
      }

      const verification = await qrCodeService.verifyQRCode(qrCodeId);

      res.json({
        success: true,
        data: {
          qrCodeId,
          valid: verification.valid,
          reason: verification.reason,
          qrCode: {
            type: qrCode.type,
            status: qrCode.status,
            expiresAt: qrCode.expiresAt,
            transportId: qrCode.transportId,
            caseId: qrCode.caseId
          }
        }
      });
    } catch (error) {
      console.error('Error verifying QR code:', error);
      res.status(500).json({
        success: false,
        message: 'Failed to verify QR code',
        error: error.message
      });
    }
  }
);

/**
 * @route   GET /api/qr-codes/stats
 * @desc    Get QR code statistics for organization
 * @access  Private (Admin)
 */
router.get('/stats',
  verifyToken,
  authorize(['admin', 'technician', 'staff', 'funeral_home_staff']),
  checkFeatureAccess('qrCodeGeneration'),
  async (req, res) => {
    try {
      console.log('QR Stats - User role:', req.user.role);
      console.log('QR Stats - User ID:', req.user._id);
      console.log('QR Stats - Organization ID:', req.user.organizationId);
      
      const { organizationId } = req.user;
      const stats = await qrCodeService.getQRCodeStats(organizationId);

      res.json({
        success: true,
        data: {
          stats,
          organizationId
        }
      });
    } catch (error) {
      console.error('Error getting QR code stats:', error);
      res.status(500).json({
        success: false,
        message: 'Failed to get QR code statistics',
        error: error.message
      });
    }
  }
);

/**
 * @route   GET /api/qr-codes/active
 * @desc    Get active QR codes for organization
 * @access  Private (Admin, Staff)
 */
router.get('/active',
  verifyToken,
  authorize(['admin', 'staff', 'technician', 'funeral_home_staff']),
  checkFeatureAccess('qrCodeGeneration'),
  async (req, res) => {
    try {
      const { organizationId } = req.user;
      const qrCodes = await QRCodeModel.findActive(organizationId);

      res.json({
        success: true,
        data: {
          qrCodes,
          count: qrCodes.length
        }
      });
    } catch (error) {
      console.error('Error getting active QR codes:', error);
      res.status(500).json({
        success: false,
        message: 'Failed to get active QR codes',
        error: error.message
      });
    }
  }
);

/**
 * @route   PUT /api/qr-codes/:qrCodeId/deactivate
 * @desc    Deactivate a QR code
 * @access  Private (Admin)
 */
router.put('/:qrCodeId/deactivate',
  verifyToken,
  authorize(['admin', 'funeral_home_staff']),
  checkFeatureAccess('qrCodeGeneration'),
  async (req, res) => {
    try {
      const { qrCodeId } = req.params;
      const { organizationId } = req.user;

      const qrCode = await QRCodeModel.findOneAndUpdate(
        { qrCodeId, organizationId },
        { isActive: false },
        { new: true }
      );

      if (!qrCode) {
        return res.status(404).json({
          success: false,
          message: 'QR code not found'
        });
      }

      res.json({
        success: true,
        message: 'QR code deactivated successfully',
        data: {
          qrCodeId,
          isActive: qrCode.isActive
        }
      });
    } catch (error) {
      console.error('Error deactivating QR code:', error);
      res.status(500).json({
        success: false,
        message: 'Failed to deactivate QR code',
        error: error.message
      });
    }
  }
);

/**
 * @route   GET /api/qr-codes/:qrCodeId
 * @desc    Get QR code details
 * @access  Private (Admin, Staff, Technician)
 */
router.get('/:qrCodeId',
  verifyToken,
  authorize(['admin', 'staff', 'technician', 'funeral_home_staff']),
  checkFeatureAccess('qrCodeGeneration'),
  async (req, res) => {
    try {
      const { qrCodeId } = req.params;
      const { organizationId } = req.user;

      const qrCode = await QRCodeModel.findOne({ 
        qrCodeId, 
        organizationId 
      }).populate('transportId', 'caseId deceasedName status');

      if (!qrCode) {
        return res.status(404).json({
          success: false,
          message: 'QR code not found'
        });
      }

      res.json({
        success: true,
        data: {
          qrCode
        }
      });
    } catch (error) {
      console.error('Error getting QR code details:', error);
      res.status(500).json({
        success: false,
        message: 'Failed to get QR code details',
        error: error.message
      });
    }
  }
);

module.exports = router;
