const express = require('express');
const router = express.Router();
const User = require('../models/User');
const Organization = require('../models/Organization');
const { verifyToken, authorize, authRateLimit } = require('../middleware/auth');
const jwt = require('jsonwebtoken');
const bcrypt = require('bcryptjs');
const crypto = require('crypto');
const { sendEmail } = require('../services/email');
const { sendSMS } = require('../services/notification');

// User registration
router.post('/register', authRateLimit, async (req, res) => {
  try {
    const {
      firstName,
      lastName,
      email,
      phone,
      password,
      role,
      organizationId,
      dateOfBirth,
      address,
      technicianInfo,
      familyInfo
    } = req.body;

    // Check if user already exists
    const existingUser = await User.findByEmail(email);
    if (existingUser) {
      return res.status(400).json({
        success: false,
        message: 'User with this email already exists'
      });
    }

    // Only allow admin role during registration
    if (role !== 'admin') {
      return res.status(400).json({
        success: false,
        message: 'Only admin accounts can be created during registration. Other user types must be added by admins.'
      });
    }

    // Validate role-specific requirements
    if (role === 'family' && !dateOfBirth) {
      return res.status(400).json({
        success: false,
        message: 'Date of birth is required for family members'
      });
    }

    // Handle organization creation for admin role
    let finalOrganizationId = organizationId;
    if (role === 'admin' && !organizationId) {
      // Create organization for admin if not provided
      const organizationName = req.body.organizationName;
      if (!organizationName) {
        return res.status(400).json({
          success: false,
          message: 'Organization name is required for admin accounts'
        });
      }

      const organization = new Organization({
        name: organizationName,
        legalName: organizationName,
        type: 'funeral_home',
        contact: {
          email: email,
          phone: phone
        },
        address: {
          street: req.body.address?.street || '123 Main St',
          city: req.body.address?.city || 'Anytown',
          state: req.body.address?.state || 'CA',
          zipCode: req.body.address?.zipCode || '12345',
          country: 'US'
        },
        business: {
          licenseNumber: req.body.licenseNumber || 'TEMP-LICENSE-001',
          licenseExpiry: req.body.licenseExpiry || new Date(Date.now() + 365 * 24 * 60 * 60 * 1000), // 1 year from now
          yearEstablished: req.body.yearEstablished || new Date().getFullYear()
        },
        subscription: {
          plan: 'basic',
          status: 'active',
          startDate: new Date()
        },
        settings: {
          notifications: {
            transportUpdates: true,
            emergencyAlerts: true,
            custodyEvents: true,
            familyUpdates: true,
            reminders: true
          },
          features: {
            blockchain: true,
            aiVoice: true,
            gpsTracking: true,
            qrScanning: true,
            voiceLogging: true
          },
          branding: {
            logo: null,
            colors: {
              primary: '#1a365d',
              secondary: '#2d3748',
              accent: '#3182ce'
            }
          },
          compliance: {
            gdpr: true,
            hipaa: true,
            auditTrails: true,
            dataRetention: 7
          }
        }
      });

      await organization.save();
      finalOrganizationId = organization._id;
    }

    // Admin role validation - organization will be created automatically
    if (role === 'admin' && !finalOrganizationId) {
      return res.status(400).json({
        success: false,
        message: 'Organization creation failed'
      });
    }

    // Create user
    const user = new User({
      firstName,
      lastName,
      email,
      phone,
      password,
      role,
      organizationId: finalOrganizationId,
      dateOfBirth,
      address,
      technicianInfo,
      familyInfo,
      createdBy: req.user?._id || null
    });

    await user.save();

    // Generate verification token
    const verificationToken = crypto.randomBytes(32).toString('hex');
    user.passwordResetToken = verificationToken;
    user.passwordResetExpires = Date.now() + 24 * 60 * 60 * 1000; // 24 hours
    await user.save();

    // Send verification email
    try {
      const baseUrl = 'https://obitix.prohiely.ltd';
      await sendEmail({
        to: user.email,
        subject: 'Welcome to Obitix - Verify Your Email',
        template: 'email-verification',
        data: {
          name: user.firstName,
          verificationUrl: `${baseUrl}/api/auth/verify-email/${verificationToken}`
        }
      });
      console.log(`Verification email sent to ${user.email}`);
    } catch (emailError) {
      console.error('Failed to send verification email:', emailError);
      // Don't fail registration if email fails
    }

    // Generate JWT token
    const token = user.generateAuthToken();

    res.status(201).json({
      success: true,
      message: 'User registered successfully',
      data: {
        user: {
          id: user._id,
          firstName: user.firstName,
          lastName: user.lastName,
          email: user.email,
          role: user.role,
          organizationId: user.organizationId
        },
        token
      }
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to register user',
      error: error.message
    });
  }
});

// User login
router.post('/login', authRateLimit, async (req, res) => {
  try {
    const { email, password, twoFactorToken } = req.body;

    // Find user by email
    const user = await User.findByEmail(email).select('+password');
    if (!user) {
      return res.status(401).json({
        success: false,
        message: 'Invalid email or password'
      });
    }

    // Check if user is active
    if (!user.isActive) {
      return res.status(401).json({
        success: false,
        message: 'Account is deactivated'
      });
    }

    // Check if user is locked
    if (user.isLocked) {
      return res.status(401).json({
        success: false,
        message: 'Account is temporarily locked due to too many failed attempts'
      });
    }

    // Verify password
    const isPasswordValid = await user.comparePassword(password);
    if (!isPasswordValid) {
      // Increment login attempts
      await user.incLoginAttempts();
      
      return res.status(401).json({
        success: false,
        message: 'Invalid email or password'
      });
    }

    // Reset login attempts on successful login
    await user.resetLoginAttempts();

    // Check two-factor authentication
    if (user.twoFactorEnabled) {
      if (!twoFactorToken) {
        return res.status(401).json({
          success: false,
          message: 'Two-factor authentication token required',
          requires2FA: true
        });
      }

      // Verify 2FA token (implementation would depend on the 2FA method)
      // For now, we'll just check if it's provided
      if (twoFactorToken.length < 6) {
        return res.status(401).json({
          success: false,
          message: 'Invalid two-factor authentication token'
        });
      }
    }

    // Update last login
    user.lastLogin = new Date();
    await user.save();

    // Generate JWT token
    const token = user.generateAuthToken();

    res.json({
      success: true,
      message: 'Login successful',
      data: {
        user: {
          id: user._id,
          firstName: user.firstName,
          lastName: user.lastName,
          email: user.email,
          role: user.role,
          organizationId: user.organizationId,
          permissions: user.permissions,
          isEmailVerified: user.isEmailVerified,
          isPhoneVerified: user.isPhoneVerified,
          twoFactorEnabled: user.twoFactorEnabled
        },
        token
      }
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Login failed',
      error: error.message
    });
  }
});

// Verify email via server link
router.get('/verify-email/:token', async (req, res) => {
  try {
    const { token } = req.params;

    const user = await User.findOne({
      passwordResetToken: token,
      passwordResetExpires: { $gt: Date.now() }
    });

    if (!user) {
      return res.status(400).send(`
        <!DOCTYPE html>
        <html>
        <head>
          <title>Email Verification Failed</title>
          <style>
            body { font-family: Arial, sans-serif; text-align: center; padding: 50px; background-color: #f8f9fa; }
            .container { max-width: 500px; margin: 0 auto; background: white; padding: 40px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
            .error { color: #dc3545; font-size: 24px; margin-bottom: 20px; }
            .message { color: #6c757d; margin-bottom: 30px; line-height: 1.6; }
            .button { background: #007bff; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; display: inline-block; }
            .icon { font-size: 48px; margin-bottom: 20px; }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="icon">❌</div>
            <div class="error">Email Verification Failed</div>
            <div class="message">
              The verification link is invalid or has expired. Please request a new verification email.
            </div>
            <a href="http://localhost:8081/login" class="button">Go to Login</a>
          </div>
        </body>
        </html>
      `);
    }

    user.isEmailVerified = true;
    user.passwordResetToken = undefined;
    user.passwordResetExpires = undefined;
    await user.save();

    res.send(`
      <!DOCTYPE html>
      <html>
      <head>
        <title>Email Verified Successfully</title>
        <style>
          body { font-family: Arial, sans-serif; text-align: center; padding: 50px; background-color: #f8f9fa; }
          .container { max-width: 500px; margin: 0 auto; background: white; padding: 40px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
          .success { color: #28a745; font-size: 24px; margin-bottom: 20px; }
          .message { color: #6c757d; margin-bottom: 30px; line-height: 1.6; }
          .button { background: #28a745; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; display: inline-block; }
          .icon { font-size: 48px; margin-bottom: 20px; }
          .countdown { color: #999; font-size: 14px; margin-top: 20px; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="icon">✅</div>
          <div class="success">Email Verified Successfully!</div>
          <div class="message">
            Hello ${user.firstName},<br><br>
            Your email address has been verified successfully. You can now log in to your Obitix account.
          </div>
          <a href="http://localhost:8081/login" class="button">Continue to Login</a>
          <div class="countdown">This page will automatically redirect to login in <span id="timer">5</span> seconds...</div>
        </div>
        
        <script>
          let timeLeft = 5;
          const timerElement = document.getElementById('timer');
          
          const countdown = setInterval(() => {
            timeLeft--;
            timerElement.textContent = timeLeft;
            
            if (timeLeft <= 0) {
              clearInterval(countdown);
              window.location.href = 'http://localhost:8081/login';
            }
          }, 1000);
        </script>
      </body>
      </html>
    `);
  } catch (error) {
    console.error('Email verification error:', error);
    res.status(500).send(`
      <!DOCTYPE html>
      <html>
      <head>
        <title>Email Verification Error</title>
        <style>
          body { font-family: Arial, sans-serif; text-align: center; padding: 50px; background-color: #f8f9fa; }
          .container { max-width: 500px; margin: 0 auto; background: white; padding: 40px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
          .error { color: #dc3545; font-size: 24px; margin-bottom: 20px; }
          .message { color: #6c757d; margin-bottom: 30px; line-height: 1.6; }
          .button { background: #007bff; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; display: inline-block; }
          .icon { font-size: 48px; margin-bottom: 20px; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="icon">⚠️</div>
          <div class="error">Verification Error</div>
          <div class="message">
            An error occurred during email verification. Please try again or contact support.
          </div>
          <a href="http://localhost:8081/login" class="button">Go to Login</a>
        </div>
      </body>
      </html>
    `);
  }
});

// Keep the POST endpoint for API calls (if needed)
router.post('/verify-email', async (req, res) => {
  try {
    const { token } = req.body;

    const user = await User.findOne({
      passwordResetToken: token,
      passwordResetExpires: { $gt: Date.now() }
    });

    if (!user) {
      return res.status(400).json({
        success: false,
        message: 'Invalid or expired verification token'
      });
    }

    user.isEmailVerified = true;
    user.passwordResetToken = undefined;
    user.passwordResetExpires = undefined;
    await user.save();

    res.json({
      success: true,
      message: 'Email verified successfully'
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Email verification failed',
      error: error.message
    });
  }
});

// Resend verification email
router.post('/resend-verification', authRateLimit, async (req, res) => {
  try {
    const { email } = req.body;

    const user = await User.findByEmail(email);
    if (!user) {
      // Don't reveal if user exists or not
      return res.json({
        success: true,
        message: 'If an account with that email exists, a verification email has been sent'
      });
    }

    if (user.isEmailVerified) {
      return res.status(400).json({
        success: false,
        message: 'Email is already verified'
      });
    }

    // Generate new verification token
    const verificationToken = crypto.randomBytes(32).toString('hex');
    user.passwordResetToken = verificationToken;
    user.passwordResetExpires = Date.now() + 24 * 60 * 60 * 1000; // 24 hours
    await user.save();

    // Send verification email
    try {
      const baseUrl = "https://obitix.prohirely.ltd";
      await sendEmail({
        to: user.email,
        subject: 'Obitix - Verify Your Email',
        template: 'email-verification',
        data: {
          name: user.firstName,
          verificationUrl: `${baseUrl}/api/auth/verify-email/${verificationToken}`
        }
      });
      console.log(`Verification email resent to ${user.email}`);
    } catch (emailError) {
      console.error('Failed to send verification email:', emailError);
    }

    res.json({
      success: true,
      message: 'If an account with that email exists, a verification email has been sent'
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to resend verification email',
      error: error.message
    });
  }
});

// Forgot password
router.post('/forgot-password', authRateLimit, async (req, res) => {
  try {
    const { email } = req.body;

    const user = await User.findByEmail(email);
    if (!user) {
      // Don't reveal if user exists or not
      return res.json({
        success: true,
        message: 'If an account with that email exists, a password reset link has been sent'
      });
    }

    // Generate reset token
    const resetToken = crypto.randomBytes(32).toString('hex');
    user.passwordResetToken = resetToken;
    user.passwordResetExpires = Date.now() + 60 * 60 * 1000; // 1 hour
    await user.save();

    // Send reset email
    try {
      const baseUrl = process.env.BACKEND_URL || 'https://obitix.prohiely.ltd';
      await sendEmail({
        to: user.email,
        subject: 'Obitix - Password Reset Request',
        template: 'password-reset',
        data: {
          name: user.firstName,
          resetUrl: `${baseUrl}/api/auth/reset-password/${resetToken}`
        }
      });
      console.log(`Password reset email sent to ${user.email}`);
    } catch (emailError) {
      console.error('Failed to send password reset email:', emailError);
      // Don't fail if email fails
    }

    res.json({
      success: true,
      message: 'If an account with that email exists, a password reset link has been sent'
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Password reset request failed',
      error: error.message
    });
  }
});

// Reset password via server link
router.get('/reset-password/:token', async (req, res) => {
  try {
    const { token } = req.params;

    const user = await User.findOne({
      passwordResetToken: token,
      passwordResetExpires: { $gt: Date.now() }
    });

    if (!user) {
      return res.status(400).send(`
        <!DOCTYPE html>
        <html>
        <head>
          <title>Password Reset Failed</title>
          <style>
            body { font-family: Arial, sans-serif; text-align: center; padding: 50px; background-color: #f8f9fa; }
            .container { max-width: 500px; margin: 0 auto; background: white; padding: 40px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
            .error { color: #dc3545; font-size: 24px; margin-bottom: 20px; }
            .message { color: #6c757d; margin-bottom: 30px; line-height: 1.6; }
            .button { background: #007bff; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; display: inline-block; }
            .icon { font-size: 48px; margin-bottom: 20px; }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="icon">❌</div>
            <div class="error">Password Reset Failed</div>
            <div class="message">
              The password reset link is invalid or has expired. Please request a new password reset email.
            </div>
            <a href="http://localhost:8081/login" class="button">Go to Login</a>
          </div>
        </body>
        </html>
      `);
    }

    // Send the password reset form
    res.send(`
      <!DOCTYPE html>
      <html>
      <head>
        <title>Reset Your Password - Obitix</title>
        <style>
          body { font-family: Arial, sans-serif; background-color: #f8f9fa; margin: 0; padding: 20px; }
          .container { max-width: 500px; margin: 0 auto; background: white; padding: 40px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
          .header { text-align: center; margin-bottom: 30px; }
          .title { color: #1a365d; font-size: 24px; margin-bottom: 10px; }
          .subtitle { color: #6c757d; font-size: 16px; }
          .form-group { margin-bottom: 20px; }
          .label { display: block; margin-bottom: 8px; color: #333; font-weight: 600; }
          .input { width: 100%; padding: 12px; border: 1px solid #ddd; border-radius: 5px; font-size: 16px; box-sizing: border-box; }
          .button { background: #1a365d; color: white; padding: 12px 24px; border: none; border-radius: 5px; font-size: 16px; cursor: pointer; width: 100%; }
          .button:hover { background: #0f2a4a; }
          .button:disabled { background: #ccc; cursor: not-allowed; }
          .message { margin-top: 20px; padding: 10px; border-radius: 5px; display: none; }
          .success { background: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
          .error { background: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
          .loading { text-align: center; color: #6c757d; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="header">
            <div class="title">Reset Your Password</div>
            <div class="subtitle">Hello ${user.firstName}, enter your new password below</div>
          </div>
          
          <form id="resetForm">
            <div class="form-group">
              <label for="password" class="label">New Password</label>
              <input type="password" id="password" class="input" placeholder="Enter your new password" required minlength="6">
            </div>
            
            <div class="form-group">
              <label for="confirmPassword" class="label">Confirm Password</label>
              <input type="password" id="confirmPassword" class="input" placeholder="Confirm your new password" required minlength="6">
            </div>
            
            <button type="submit" class="button" id="submitBtn">Reset Password</button>
          </form>
          
          <div id="message" class="message"></div>
        </div>
        
        <script>
          const form = document.getElementById('resetForm');
          const message = document.getElementById('message');
          const submitBtn = document.getElementById('submitBtn');
          
          form.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const password = document.getElementById('password').value;
            const confirmPassword = document.getElementById('confirmPassword').value;
            
            if (password !== confirmPassword) {
              showMessage('Passwords do not match', 'error');
              return;
            }
            
            if (password.length < 6) {
              showMessage('Password must be at least 6 characters long', 'error');
              return;
            }
            
            submitBtn.disabled = true;
            submitBtn.textContent = 'Resetting Password...';
            
            try {
              const response = await fetch('/api/auth/reset-password/${token}', {
                method: 'POST',
                headers: {
                  'Content-Type': 'application/json',
                },
                body: JSON.stringify({ password })
              });
              
              const result = await response.json();
              
              if (result.success) {
                showMessage('Password reset successfully! Redirecting to login...', 'success');
                setTimeout(() => {
                  window.location.href = 'http://localhost:8081/login';
                }, 2000);
              } else {
                showMessage(result.message || 'Password reset failed', 'error');
                submitBtn.disabled = false;
                submitBtn.textContent = 'Reset Password';
              }
            } catch (error) {
              showMessage('An error occurred. Please try again.', 'error');
              submitBtn.disabled = false;
              submitBtn.textContent = 'Reset Password';
            }
          });
          
          function showMessage(text, type) {
            message.textContent = text;
            message.className = 'message ' + type;
            message.style.display = 'block';
          }
        </script>
      </body>
      </html>
    `);
  } catch (error) {
    console.error('Password reset error:', error);
    res.status(500).send(`
      <!DOCTYPE html>
      <html>
      <head>
        <title>Password Reset Error</title>
        <style>
          body { font-family: Arial, sans-serif; text-align: center; padding: 50px; background-color: #f8f9fa; }
          .container { max-width: 500px; margin: 0 auto; background: white; padding: 40px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
          .error { color: #dc3545; font-size: 24px; margin-bottom: 20px; }
          .message { color: #6c757d; margin-bottom: 30px; line-height: 1.6; }
          .button { background: #007bff; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; display: inline-block; }
          .icon { font-size: 48px; margin-bottom: 20px; }
        </style>
      </head>
      <body>
        <div class="container">
          <div class="icon">⚠️</div>
          <div class="error">Password Reset Error</div>
          <div class="message">
            An error occurred during password reset. Please try again or contact support.
          </div>
          <a href="http://localhost:8081/login" class="button">Go to Login</a>
        </div>
      </body>
      </html>
    `);
  }
});

// Reset password (POST endpoint for API calls)
router.post('/reset-password', async (req, res) => {
  try {
    const { token, password } = req.body;

    const user = await User.findOne({
      passwordResetToken: token,
      passwordResetExpires: { $gt: Date.now() }
    });

    if (!user) {
      return res.status(400).json({
        success: false,
        message: 'Invalid or expired reset token'
      });
    }

    // Hash new password
    const salt = await bcrypt.genSalt(10);
    user.password = await bcrypt.hash(password, salt);
    user.passwordResetToken = undefined;
    user.passwordResetExpires = undefined;
    await user.save();

    res.json({
      success: true,
      message: 'Password reset successfully'
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Password reset failed',
      error: error.message
    });
  }
});

// Change password
router.post('/change-password', verifyToken, async (req, res) => {
  try {
    const { currentPassword, newPassword } = req.body;

    const user = await User.findById(req.user._id).select('+password');

    // Verify current password
    const isCurrentPasswordValid = await user.comparePassword(currentPassword);
    if (!isCurrentPasswordValid) {
      return res.status(400).json({
        success: false,
        message: 'Current password is incorrect'
      });
    }

    // Update password
    user.password = newPassword;
    await user.save();

    res.json({
      success: true,
      message: 'Password changed successfully'
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Password change failed',
      error: error.message
    });
  }
});

// Get current user
router.get('/me', verifyToken, async (req, res) => {
  try {
    const user = await User.findById(req.user._id)
      .populate('organizationId', 'name type status')
      .select('-password');

    if (!user) {
      return res.status(404).json({
        success: false,
        message: 'User not found'
      });
    }

    res.json({
      success: true,
      data: user
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to fetch user data',
      error: error.message
    });
  }
});

// Update user profile
router.put('/profile', verifyToken, async (req, res) => {
  try {
    const {
      firstName,
      lastName,
      phone,
      address,
      preferences,
      technicianInfo,
      familyInfo
    } = req.body;

    const user = await User.findById(req.user._id);
    if (!user) {
      return res.status(404).json({
        success: false,
        message: 'User not found'
      });
    }

    // Update allowed fields
    if (firstName) user.firstName = firstName;
    if (lastName) user.lastName = lastName;
    if (phone) user.phone = phone;
    if (address) user.address = address;
    if (preferences) user.preferences = { ...user.preferences, ...preferences };
    if (technicianInfo) user.technicianInfo = { ...user.technicianInfo, ...technicianInfo };
    if (familyInfo) user.familyInfo = { ...user.familyInfo, ...familyInfo };

    user.updatedBy = req.user._id;
    await user.save();

    res.json({
      success: true,
      message: 'Profile updated successfully',
      data: user
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to update profile',
      error: error.message
    });
  }
});

// Logout
router.post('/logout', verifyToken, async (req, res) => {
  try {
    // In a more sophisticated implementation, you might want to:
    // 1. Add the token to a blacklist
    // 2. Update user's last logout time
    // 3. Clear session data

    res.json({
      success: true,
      message: 'Logged out successfully'
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Logout failed',
      error: error.message
    });
  }
});

// Refresh token
router.post('/refresh-token', verifyToken, async (req, res) => {
  try {
    const user = await User.findById(req.user._id);
    if (!user) {
      return res.status(401).json({
        success: false,
        message: 'User not found'
      });
    }

    // Generate new token
    const token = user.generateAuthToken();

    res.json({
      success: true,
      data: {
        token
      }
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Token refresh failed',
      error: error.message
    });
  }
});

// Verify phone number
router.post('/verify-phone', verifyToken, async (req, res) => {
  try {
    const { phone, verificationCode } = req.body;

    // In a real implementation, you would verify the code against what was sent
    // For now, we'll just check if a code was provided
    if (!verificationCode || verificationCode.length !== 6) {
      return res.status(400).json({
        success: false,
        message: 'Invalid verification code'
      });
    }

    const user = await User.findById(req.user._id);
    user.phone = phone;
    user.isPhoneVerified = true;
    await user.save();

    res.json({
      success: true,
      message: 'Phone number verified successfully'
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Phone verification failed',
      error: error.message
    });
  }
});

// Enable/disable two-factor authentication
router.post('/2fa', verifyToken, async (req, res) => {
  try {
    const { enable, method = 'totp' } = req.body;

    const user = await User.findById(req.user._id);
    
    if (enable) {
      // Generate 2FA secret
      const secret = crypto.randomBytes(32).toString('hex');
      user.twoFactorSecret = secret;
      user.twoFactorEnabled = true;
      
      // In a real implementation, you would generate a QR code for TOTP apps
      await user.save();

      res.json({
        success: true,
        message: 'Two-factor authentication enabled',
        data: {
          secret,
          qrCode: `otpauth://totp/Obitix:${user.email}?secret=${secret}&issuer=Obitix`
        }
      });
    } else {
      user.twoFactorEnabled = false;
      user.twoFactorSecret = undefined;
      await user.save();

      res.json({
        success: true,
        message: 'Two-factor authentication disabled'
      });
    }
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Two-factor authentication update failed',
      error: error.message
    });
  }
});

// Organization registration
router.post('/register-organization', authRateLimit, async (req, res) => {
  try {
    const {
      name,
      legalName,
      type,
      contact,
      address,
      business
    } = req.body;

    // Check if organization already exists
    const existingOrg = await Organization.findByLicense(business.licenseNumber);
    if (existingOrg) {
      return res.status(400).json({
        success: false,
        message: 'Organization with this license number already exists'
      });
    }

    // Create organization
    const organization = new Organization({
      name,
      legalName,
      type,
      contact,
      address,
      business
    });

    await organization.save();

    res.status(201).json({
      success: true,
      message: 'Organization registered successfully',
      data: {
        id: organization._id,
        name: organization.name,
        status: organization.status
      }
    });
  } catch (error) {
    res.status(500).json({
      success: false,
      message: 'Failed to register organization',
      error: error.message
    });
  }
});

module.exports = router;
