const mongoose = require('mongoose');

const organizationSchema = new mongoose.Schema({
  // Basic Information
  name: {
    type: String,
    required: [true, 'Organization name is required'],
    trim: true,
    maxlength: [100, 'Organization name cannot exceed 100 characters']
  },
  legalName: {
    type: String,
    required: [true, 'Legal name is required'],
    trim: true
  },
  type: {
    type: String,
    enum: ['funeral_home', 'crematory', 'cemetery', 'transport_company', 'corporate'],
    required: true
  },
  status: {
    type: String,
    enum: ['active', 'inactive', 'suspended', 'pending_verification'],
    default: 'pending_verification'
  },
  
  // Contact Information
  contact: {
    email: {
      type: String,
      required: [true, 'Email is required'],
      lowercase: true,
      trim: true
    },
    phone: {
      type: String,
      required: [true, 'Phone number is required']
    },
    website: String,
    fax: String
  },
  
  // Address Information
  address: {
    street: {
      type: String,
      required: [true, 'Street address is required']
    },
    city: {
      type: String,
      required: [true, 'City is required']
    },
    state: {
      type: String,
      required: [true, 'State is required']
    },
    zipCode: {
      type: String,
      required: [true, 'ZIP code is required']
    },
    country: {
      type: String,
      default: 'US'
    },
    coordinates: {
      latitude: Number,
      longitude: Number
    }
  },
  
  // Business Information
  business: {
    licenseNumber: {
      type: String,
      required: [true, 'License number is required']
    },
    licenseExpiry: {
      type: Date,
      required: [true, 'License expiry date is required']
    },
    taxId: String,
    dba: String, // Doing Business As
    yearEstablished: Number,
    businessHours: {
      monday: { open: String, close: String },
      tuesday: { open: String, close: String },
      wednesday: { open: String, close: String },
      thursday: { open: String, close: String },
      friday: { open: String, close: String },
      saturday: { open: String, close: String },
      sunday: { open: String, close: String }
    },
    services: [{
      type: String,
      enum: [
        'traditional_funeral',
        'cremation',
        'burial',
        'memorial_service',
        'pre_planning',
        'transportation',
        'embalming',
        'viewing',
        'repatriation',
        'green_burial',
        'aquamation'
      ]
    }]
  },
  
  // Subscription and Billing
  subscription: {
    plan: {
      type: String,
      enum: ['basic', 'professional', 'enterprise', 'custom'],
      default: 'basic'
    },
    status: {
      type: String,
      enum: ['active', 'trial', 'past_due', 'cancelled', 'suspended'],
      default: 'trial'
    },
    startDate: {
      type: Date,
      default: Date.now
    },
    endDate: Date,
    monthlyLimit: {
      type: Number,
      default: 50
    },
    currentUsage: {
      type: Number,
      default: 0
    },
    stripeCustomerId: String,
    stripeSubscriptionId: String
  },
  
  // Settings and Configuration
  settings: {
    notifications: {
      email: { type: Boolean, default: true },
      sms: { type: Boolean, default: true },
      push: { type: Boolean, default: true },
      voice: { type: Boolean, default: false }
    },
    features: {
      aiVoiceAssistant: { type: Boolean, default: true },
      blockchainTracking: { type: Boolean, default: true },
      realTimeGPS: { type: Boolean, default: true },
      familyPortal: { type: Boolean, default: true },
      auditLogging: { type: Boolean, default: true },
      complianceReporting: { type: Boolean, default: true }
    },
    branding: {
      logo: String,
      primaryColor: {
        type: String,
        default: '#1a365d'
      },
      secondaryColor: {
        type: String,
        default: '#2d3748'
      },
      customDomain: String,
      whiteLabel: {
        type: Boolean,
        default: false
      }
    },
    compliance: {
      hipaaCompliant: { type: Boolean, default: true },
      gdprCompliant: { type: Boolean, default: true },
      stateRegulations: [String],
      auditFrequency: {
        type: String,
        enum: ['monthly', 'quarterly', 'annually'],
        default: 'quarterly'
      }
    }
  },
  
  // Staff and Personnel
  staff: [{
    user: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'User'
    },
    role: {
      type: String,
      enum: ['owner', 'manager', 'staff', 'technician'],
      required: true
    },
    permissions: [String],
    isActive: {
      type: Boolean,
      default: true
    },
    joinedAt: {
      type: Date,
      default: Date.now
    }
  }],
  
  // Equipment and Vehicles
  equipment: [{
    type: {
      type: String,
      enum: ['hearse', 'transport_vehicle', 'refrigeration_unit', 'other']
    },
    name: String,
    description: String,
    licensePlate: String,
    gpsEnabled: {
      type: Boolean,
      default: false
    },
    isActive: {
      type: Boolean,
      default: true
    }
  }],
  
  // Partners and Affiliations
  partners: [{
    organization: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'Organization'
    },
    relationship: {
      type: String,
      enum: ['crematory', 'cemetery', 'transport', 'supplier', 'other']
    },
    isActive: {
      type: Boolean,
      default: true
    }
  }],
  
  // Statistics and Analytics
  statistics: {
    totalTransports: {
      type: Number,
      default: 0
    },
    monthlyTransports: {
      type: Number,
      default: 0
    },
    averageRating: {
      type: Number,
      default: 0
    },
    totalReviews: {
      type: Number,
      default: 0
    },
    lastActivity: Date
  },
  
  // Documents and Certifications
  documents: [{
    type: {
      type: String,
      enum: [
        'business_license',
        'insurance_certificate',
        'tax_exemption',
        'accreditation',
        'other'
      ]
    },
    name: String,
    url: String,
    ipfsHash: String,
    expiryDate: Date,
    isVerified: {
      type: Boolean,
      default: false
    },
    uploadedAt: {
      type: Date,
      default: Date.now
    }
  }],
  
  // Audit and Compliance
  audit: {
    lastAuditDate: Date,
    nextAuditDate: Date,
    complianceScore: {
      type: Number,
      min: 0,
      max: 100,
      default: 0
    },
    violations: [{
      type: String,
      description: String,
      date: Date,
      resolved: {
        type: Boolean,
        default: false
      },
      resolutionDate: Date
    }]
  },
  
  // API and Integration
  integrations: {
    fhms: [{
      name: String,
      apiKey: String,
      isActive: {
        type: Boolean,
        default: false
      },
      lastSync: Date
    }],
    webhooks: [{
      url: String,
      events: [String],
      isActive: {
        type: Boolean,
        default: false
      }
    }]
  },
  
  // Notes and Comments
  notes: [{
    content: String,
    author: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'User'
    },
    timestamp: {
      type: Date,
      default: Date.now
    },
    isPrivate: {
      type: Boolean,
      default: false
    }
  }],
  
  // Timestamps
  createdAt: {
    type: Date,
    default: Date.now
  },
  updatedAt: {
    type: Date,
    default: Date.now
  },
  verifiedAt: Date,
  verifiedBy: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User'
  }
}, {
  timestamps: true,
  toJSON: { virtuals: true },
  toObject: { virtuals: true }
});

// Virtual for full address
organizationSchema.virtual('fullAddress').get(function() {
  return `${this.address.street}, ${this.address.city}, ${this.address.state} ${this.address.zipCode}`;
});

// Virtual for subscription status
organizationSchema.virtual('isSubscriptionActive').get(function() {
  return ['active', 'trial'].includes(this.subscription.status);
});

// Virtual for license status
organizationSchema.virtual('isLicenseValid').get(function() {
  return this.business.licenseExpiry > new Date();
});

// Indexes
organizationSchema.index({ name: 1 });
organizationSchema.index({ 'business.licenseNumber': 1 });
organizationSchema.index({ status: 1 });
organizationSchema.index({ 'subscription.status': 1 });
organizationSchema.index({ 'address.state': 1, 'address.city': 1 });

// Pre-save middleware to update updatedAt
organizationSchema.pre('save', function(next) {
  this.updatedAt = Date.now();
  next();
});

// Static method to find active organizations
organizationSchema.statics.findActive = function() {
  return this.find({ status: 'active' });
};

// Static method to find by license number
organizationSchema.statics.findByLicense = function(licenseNumber) {
  return this.findOne({ 'business.licenseNumber': licenseNumber });
};

// Static method to find by location
organizationSchema.statics.findByLocation = function(state, city) {
  return this.find({
    'address.state': state,
    'address.city': city,
    status: 'active'
  });
};

// Instance method to update statistics
organizationSchema.methods.updateStatistics = function() {
  // This would typically be called after transport completion
  this.statistics.totalTransports += 1;
  this.statistics.lastActivity = new Date();
  return this.save();
};

// Instance method to check subscription limits
organizationSchema.methods.checkSubscriptionLimit = function() {
  return this.subscription.currentUsage < this.subscription.monthlyLimit;
};

module.exports = mongoose.model('Organization', organizationSchema);
