const mongoose = require('mongoose');

const notificationSchema = new mongoose.Schema({
  // User who will receive the notification
  userId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
    required: true,
    index: true
  },

  // Notification type
  type: {
    type: String,
    enum: ['status_update', 'location_update', 'delay_alert', 'emergency_alert', 'general'],
    required: true
  },

  // Notification content
  title: {
    type: String,
    required: true
  },

  message: {
    type: String,
    required: true
  },

  // Additional data for the notification
  data: {
    transportId: mongoose.Schema.Types.ObjectId,
    oldStatus: String,
    newStatus: String,
    currentLocation: {
      coordinates: {
        latitude: Number,
        longitude: Number
      },
      address: String,
      timestamp: Date
    },
    estimatedDeliveryTime: Date,
    emergencyType: String,
    emergencyMessage: String
  },

  // Priority level
  priority: {
    type: String,
    enum: ['low', 'normal', 'high', 'urgent'],
    default: 'normal'
  },

  // Read status
  isRead: {
    type: Boolean,
    default: false
  },

  // Delivery status
  isDelivered: {
    type: Boolean,
    default: false
  },

  // Delivery method
  deliveryMethod: {
    type: String,
    enum: ['push', 'email', 'sms', 'in_app'],
    default: 'in_app'
  },

  // Scheduled delivery time (for delayed notifications)
  scheduledAt: Date,

  // Actual delivery time
  deliveredAt: Date,

  // Read time
  readAt: Date,

  // Organization context
  organizationId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Organization',
    required: true
  },

  // Metadata
  metadata: {
    source: String, // 'system', 'admin', 'technician'
    sourceUserId: mongoose.Schema.Types.ObjectId,
    tags: [String],
    category: String
  }
}, {
  timestamps: true
});

// Indexes for efficient querying
notificationSchema.index({ userId: 1, isRead: 1, createdAt: -1 });
notificationSchema.index({ organizationId: 1, type: 1, createdAt: -1 });
notificationSchema.index({ scheduledAt: 1, isDelivered: 1 });

// Pre-save middleware to set organization if not provided
notificationSchema.pre('save', async function(next) {
  if (!this.organizationId) {
    try {
      const User = require('./User');
      const user = await User.findById(this.userId);
      if (user) {
        this.organizationId = user.organizationId;
      }
    } catch (error) {
      console.error('Error setting organization ID for notification:', error);
    }
  }
  next();
});

// Instance methods
notificationSchema.methods.markAsRead = function() {
  this.isRead = true;
  this.readAt = new Date();
  return this.save();
};

notificationSchema.methods.markAsDelivered = function() {
  this.isDelivered = true;
  this.deliveredAt = new Date();
  return this.save();
};

// Static methods
notificationSchema.statics.getUnreadCount = function(userId) {
  return this.countDocuments({ userId, isRead: false });
};

notificationSchema.statics.getUnreadNotifications = function(userId, limit = 50) {
  return this.find({ userId, isRead: false })
    .sort({ createdAt: -1 })
    .limit(limit)
    .populate('data.transportId', 'caseId deceased');
};

notificationSchema.statics.markAllAsRead = function(userId) {
  return this.updateMany(
    { userId, isRead: false },
    { isRead: true, readAt: new Date() }
  );
};

notificationSchema.statics.getPendingDeliveries = function() {
  return this.find({
    isDelivered: false,
    scheduledAt: { $lte: new Date() }
  }).populate('userId', 'email phone pushToken');
};

module.exports = mongoose.model('Notification', notificationSchema);
