const Subscription = require('../models/Subscription');

/**
 * Middleware to check if the organization has access to a specific feature
 * @param {string} featureName - The feature to check access for
 * @param {boolean} requireActive - Whether to require an active subscription (default: true)
 */
const checkFeatureAccess = (featureName, requireActive = true) => {
  return async (req, res, next) => {
    try {
      // Skip subscription check for family users
      if (req.user.role === 'family') {
        return next();
      }

      // Get organization subscription
      const subscription = await Subscription.findOne({ 
        organizationId: req.user.organizationId 
      });

      // If no subscription and we require active subscription
      if (!subscription && requireActive) {
        return res.status(403).json({
          success: false,
          message: 'Active subscription required to access this feature',
          error: 'SUBSCRIPTION_REQUIRED'
        });
      }

      // If subscription exists but is not active and we require active subscription
      if (subscription && requireActive && subscription.status !== 'active') {
        return res.status(403).json({
          success: false,
          message: 'Active subscription required to access this feature',
          error: 'SUBSCRIPTION_INACTIVE',
          subscriptionStatus: subscription.status
        });
      }

      // If no subscription and we don't require active subscription, allow access
      if (!subscription && !requireActive) {
        return next();
      }

      // Check if the feature is available in the subscription
      if (subscription && !subscription.hasFeature(featureName)) {
        return res.status(403).json({
          success: false,
          message: `Feature '${featureName}' is not available in your current subscription tier`,
          error: 'FEATURE_NOT_AVAILABLE',
          currentTier: subscription.tier,
          requiredTier: getRequiredTierForFeature(featureName)
        });
      }

      // Add subscription info to request for use in controllers
      req.subscription = subscription;
      next();
    } catch (error) {
      console.error('Error checking feature access:', error);
      res.status(500).json({
        success: false,
        message: 'Error checking subscription access'
      });
    }
  };
};

/**
 * Middleware to check if the organization is within usage limits
 * @param {string} limitName - The limit to check
 * @param {number} currentValue - Current usage value
 */
const checkUsageLimit = (limitName) => {
  return async (req, res, next) => {
    try {
      // Skip limit check for family users
      if (req.user.role === 'family') {
        return next();
      }

      const subscription = await Subscription.findOne({ 
        organizationId: req.user.organizationId 
      });

      if (!subscription) {
        return res.status(403).json({
          success: false,
          message: 'Active subscription required',
          error: 'SUBSCRIPTION_REQUIRED'
        });
      }

      // Get current usage (this would need to be calculated based on your data)
      const currentUsage = await getCurrentUsage(req.user.organizationId, limitName);
      
      if (!subscription.checkLimit(limitName, currentUsage)) {
        return res.status(403).json({
          success: false,
          message: `Usage limit exceeded for ${limitName}`,
          error: 'USAGE_LIMIT_EXCEEDED',
          currentUsage,
          limit: subscription.limits[limitName]
        });
      }

      next();
    } catch (error) {
      console.error('Error checking usage limit:', error);
      res.status(500).json({
        success: false,
        message: 'Error checking usage limits'
      });
    }
  };
};

/**
 * Get the required tier for a specific feature
 */
const getRequiredTierForFeature = (featureName) => {
  const featureTiers = {
    // Tier 1: Basic Features (available in all tiers)
    caseTracking: 'basic',
    qrCodeGeneration: 'basic',
    mobileAppAccess: 'basic',
    pdfReports: 'basic',
    basicEmailSupport: 'basic',
    
    // Tier 2: Professional Features
    voiceLoggingAssistant: 'basic', // Changed to basic tier
    familyWebPortal: 'professional',
    automatedNotifications: 'professional',
    multiLocationManagement: 'professional',
    analyticsDashboard: 'professional',
    prioritySupport: 'professional',
    
    // Tier 3: Enterprise Features
    apiIntegrations: 'enterprise',
    blockchainVerification: 'enterprise',
    customBrandedQrCodes: 'enterprise',
    customizableForms: 'enterprise',
    aiVoiceAgent: 'basic', // Changed to basic tier for family access
    dedicatedAccountManager: 'enterprise',
    complianceReporting: 'enterprise'
  };

  return featureTiers[featureName] || 'basic';
};

/**
 * Get current usage for a specific limit
 * This is a placeholder - implement based on your actual data structure
 */
const getCurrentUsage = async (organizationId, limitName) => {
  // This would need to be implemented based on your actual data
  // For now, return 0 as placeholder
  switch (limitName) {
    case 'transportsPerMonth':
      // TODO: Count transports created this month
      return 0;
    case 'technicians':
      // TODO: Count users with technician role
      return 0;
    case 'locations':
      // TODO: Count locations
      return 0;
    case 'apiCallsPerMonth':
      // TODO: Track API calls
      return 0;
    case 'storageGB':
      // TODO: Calculate storage usage
      return 0;
    default:
      return 0;
  }
};

/**
 * Middleware to require a minimum subscription tier
 */
const requireTier = (minimumTier) => {
  const tierOrder = ['basic', 'professional', 'enterprise'];
  
  return async (req, res, next) => {
    try {
      // Skip tier check for family users
      if (req.user.role === 'family') {
        return next();
      }

      const subscription = await Subscription.findOne({ 
        organizationId: req.user.organizationId 
      });

      if (!subscription) {
        return res.status(403).json({
          success: false,
          message: 'Active subscription required',
          error: 'SUBSCRIPTION_REQUIRED'
        });
      }

      const currentTierIndex = tierOrder.indexOf(subscription.tier);
      const requiredTierIndex = tierOrder.indexOf(minimumTier);

      if (currentTierIndex < requiredTierIndex) {
        return res.status(403).json({
          success: false,
          message: `This feature requires ${minimumTier} tier or higher`,
          error: 'INSUFFICIENT_TIER',
          currentTier: subscription.tier,
          requiredTier: minimumTier
        });
      }

      req.subscription = subscription;
      next();
    } catch (error) {
      console.error('Error checking tier requirement:', error);
      res.status(500).json({
        success: false,
        message: 'Error checking subscription tier'
      });
    }
  };
};

module.exports = {
  checkFeatureAccess,
  checkUsageLimit,
  requireTier,
  getRequiredTierForFeature
};
