const rateLimit = require('express-rate-limit');

// MongoDB-only rate limiting configuration

// Basic rate limiter configuration (in-memory)
const rateLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 100, // limit each IP to 100 requests per windowMs
  message: {
    success: false,
    message: 'Too many requests from this IP, please try again later.'
  },
  standardHeaders: true,
  legacyHeaders: false,
  // Skip successful requests
  skipSuccessfulRequests: false,
  // Skip failed requests
  skipFailedRequests: false,
  // Key generator
  keyGenerator: (req) => {
    return req.ip; // Use IP address as key
  },
  // Handler for when limit is exceeded
  handler: (req, res) => {
    res.status(429).json({
      success: false,
      message: 'Too many requests from this IP, please try again later.',
      retryAfter: Math.ceil(1 * 60 / 1000) // 15 minutes in seconds
    });
  }
});

// Specific rate limiters for different endpoints
const authRateLimiter = (req, res, next) => {
  // Rate limiting disabled for development
  next();
};

const apiRateLimiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 1000, // limit each IP to 1000 requests per windowMs
  message: {
    success: false,
    message: 'API rate limit exceeded, please try again later.'
  },
  standardHeaders: true,
  legacyHeaders: false,
  keyGenerator: (req) => req.ip,
  handler: (req, res) => {
    res.status(429).json({
      success: false,
      message: 'API rate limit exceeded, please try again later.',
      retryAfter: Math.ceil(15 * 60 / 1000)
    });
  }
});

// File upload rate limiter
const uploadRateLimiter = rateLimit({
  windowMs: 60 * 60 * 1000, // 1 hour
  max: 10, // limit each IP to 10 uploads per hour
  message: {
    success: false,
    message: 'Too many file uploads, please try again later.'
  },
  standardHeaders: true,
  legacyHeaders: false,
  keyGenerator: (req) => req.ip,
  handler: (req, res) => {
    res.status(429).json({
      success: false,
      message: 'Too many file uploads, please try again later.',
      retryAfter: Math.ceil(60 * 60 / 1000)
    });
  }
});

// Subscription rate limiter
const subscriptionRateLimiter = rateLimit({
  windowMs: 60 * 60 * 1000, // 1 hour
  max: 10, // limit each IP to 10 subscription operations per hour
  message: {
    success: false,
    message: 'Too many subscription operations, please try again later.'
  },
  standardHeaders: true,
  legacyHeaders: false,
  keyGenerator: (req) => req.ip,
  handler: (req, res) => {
    res.status(429).json({
      success: false,
      message: 'Too many subscription operations, please try again later.',
      retryAfter: Math.ceil(60 * 60 / 1000)
    });
  }
});

module.exports = {
  rateLimiter,
  authRateLimiter,
  apiRateLimiter,
  uploadRateLimiter,
  subscriptionRateLimiter
};
