const QRCode = require('qrcode');
const mongoose = require('mongoose');
const QRCodeModel = require('../models/QRCode');
const Transport = require('../models/Transport');
const { blockchainService } = require('./blockchain');
const { uploadToIPFS } = require('./ipfs');

class QRCodeService {
  /**
   * Generate QR codes for a transport case
   * @param {string} transportId - Transport ID
   * @param {string} organizationId - Organization ID
   * @param {string} generatedBy - User ID who generated the codes
   * @returns {Promise<Array>} Array of generated QR codes
   */
  async generateTransportQRCodes(transportId, organizationId, generatedBy) {
    try {
      // Validate inputs
      if (!transportId || !organizationId || !generatedBy) {
        throw new Error('Missing required parameters for QR code generation');
      }

      // Get transport details
      const transport = await Transport.findById(transportId);
      if (!transport) {
        throw new Error('Transport not found');
      }

      // Check if QR code already exists for this transport
      const existingQRCode = await QRCodeModel.findOne({ transportId, isActive: true });
      if (existingQRCode) {
        throw new Error('QR code already exists for this transport');
      }

      // Create single QR code for the transport (not multiple types)
      const qrCode = new QRCodeModel({
        transportId,
        caseId: transport.caseId || `CASE-${Date.now()}`,
        type: 'pickup', // Use valid enum value
        location: {
          type: 'Point',
          coordinates: (() => {
            // Handle different coordinate formats
            if (Array.isArray(transport.pickupLocation?.coordinates)) {
              return transport.pickupLocation.coordinates;
            }
            if (transport.pickupLocation?.coordinates && typeof transport.pickupLocation.coordinates === 'string') {
              try {
                const parsed = JSON.parse(transport.pickupLocation.coordinates);
                return Array.isArray(parsed) ? parsed : [0, 0];
              } catch (e) {
                console.warn('Failed to parse coordinates string:', transport.pickupLocation.coordinates);
                return [0, 0];
              }
            }
            // Default coordinates if none available
            return [0, 0];
          })(),
          address: transport.pickupLocation?.address?.street || '',
          facility: transport.pickupLocation?.name || ''
        },
        organizationId,
        metadata: {
          generatedBy
        },
        expiresAt: new Date(Date.now() + 7 * 24 * 60 * 60 * 1000) // 7 days
      });

      // Generate QR data with full transport information
      try {
        qrCode.qrData = await qrCode.generateQRData();
      } catch (qrDataError) {
        console.error('Error generating QR data:', qrDataError);
        // Fallback to basic QR data
        qrCode.qrData = JSON.stringify({
          qrCodeId: qrCode.qrCodeId,
          transportId: qrCode.transportId.toString(),
          caseId: qrCode.caseId,
          type: qrCode.type,
          timestamp: qrCode.metadata.generatedAt,
          organizationId: qrCode.organizationId.toString()
        });
      }

      // Generate QR code image
      let qrCodeImage;
      try {
        console.log('Generating QR code image for data:', qrCode.qrData.substring(0, 100) + '...');
        qrCodeImage = await this.generateQRCodeImage(qrCode.qrData, qrCode.qrCodeId);
        console.log('QR code image generated successfully, length:', qrCodeImage?.length || 0);
        console.log('QR code image starts with:', qrCodeImage?.substring(0, 50) || 'no image');
      } catch (imageError) {
        console.error('Error generating QR code image:', imageError);
        throw new Error('Failed to generate QR code image');
      }

      // Save QR code
      try {
        await qrCode.save();
        console.log('QR code saved successfully:', qrCode.qrCodeId);
      } catch (saveError) {
        console.error('Error saving QR code:', saveError);
        throw new Error('Failed to save QR code');
      }

      // Add to blockchain if enterprise subscription (non-blocking)
      try {
        if (await this.shouldUseBlockchain(organizationId)) {
          await this.addToBlockchain(qrCode);
        }
      } catch (blockchainError) {
        console.error('Error adding to blockchain:', blockchainError);
        // Don't fail the entire process for blockchain errors
      }

      return [{
        qrCodeId: qrCode.qrCodeId,
        type: qrCode.type,
        qrCodeImage,
        qrData: qrCode.qrData,
        expiresAt: qrCode.expiresAt,
        transport: {
          caseId: transport.caseId,
          deceasedName: transport.deceasedName || (transport.deceased ? `${transport.deceased.firstName || ''} ${transport.deceased.lastName || ''}`.trim() : 'Unknown'),
          status: transport.status || 'scheduled',
          pickupLocation: transport.pickupLocation,
          deliveryLocation: transport.deliveryLocation
        }
      }];
    } catch (error) {
      console.error('Error generating QR codes:', error);
      throw error;
    }
  }

  /**
   * Auto-generate QR code when transport is created
   * @param {string} transportId - Transport ID
   * @param {string} organizationId - Organization ID
   * @param {string} generatedBy - User ID who generated the codes
   * @returns {Promise<Object>} Generated QR code
   */
  async autoGenerateQRCodeForTransport(transportId, organizationId, generatedBy) {
    try {
      console.log(`Auto-generating QR code for transport: ${transportId}`);
      
      // Validate inputs
      if (!transportId || !organizationId || !generatedBy) {
        console.error('Missing required parameters for QR code generation');
        return null;
      }

      // Check if transport exists
      const transport = await Transport.findById(transportId);
      if (!transport) {
        console.error(`Transport not found: ${transportId}`);
        return null;
      }

      // Check if QR code already exists
      const existingQRCode = await QRCodeModel.findOne({ transportId, isActive: true });
      if (existingQRCode) {
        console.log(`QR code already exists for transport: ${transportId}`);
        return {
          qrCodeId: existingQRCode.qrCodeId,
          type: existingQRCode.type,
          qrCodeImage: await this.generateQRCodeImage(existingQRCode.qrData, existingQRCode.qrCodeId),
          qrData: existingQRCode.qrData,
          expiresAt: existingQRCode.expiresAt
        };
      }

      const qrCodes = await this.generateTransportQRCodes(transportId, organizationId, generatedBy);
      return qrCodes[0]; // Return the single QR code
    } catch (error) {
      console.error('Error auto-generating QR code:', error);
      // Don't throw error to avoid breaking transport creation
      return null;
    }
  }

  /**
   * Generate QR code image
   * @param {string} data - Data to encode
   * @param {string} qrCodeId - Unique QR code ID
   * @returns {Promise<string>} Base64 encoded QR code image
   */
  async generateQRCodeImage(data, qrCodeId) {
    try {
      const options = {
        errorCorrectionLevel: 'H',
        type: 'image/png',
        quality: 0.92,
        margin: 1,
        color: {
          dark: '#000000',
          light: '#FFFFFF'
        },
        width: 300
      };

      const qrCodeImage = await QRCode.toDataURL(data, options);
      return qrCodeImage;
    } catch (error) {
      console.error('Error generating QR code image:', error);
      throw error;
    }
  }

  /**
   * Scan QR code and record the event
   * @param {string} qrCodeId - QR code ID
   * @param {Object} scanData - Scan event data
   * @returns {Promise<Object>} Scan result
   */
  async scanQRCode(qrCodeId, scanData) {
    try {
      // Find QR code
      const qrCode = await QRCodeModel.findOne({ qrCodeId, isActive: true });
      if (!qrCode) {
        throw new Error('QR code not found or inactive');
      }

      // Verify QR code
      const verification = qrCode.verifyQRCode();
      if (!verification.valid) {
        throw new Error(verification.reason);
      }

      // Process photos if provided
      let processedPhotos = [];
      if (scanData.photos && scanData.photos.length > 0) {
        processedPhotos = await this.processPhotos(scanData.photos);
      }

      // Process voice logs if provided
      let processedVoiceLogs = [];
      if (scanData.voiceLogs && scanData.voiceLogs.length > 0) {
        processedVoiceLogs = await this.processVoiceLogs(scanData.voiceLogs);
      }

      // Create scan event
      const scanEvent = {
        scannedBy: scanData.scannedBy,
        scannedAt: new Date(),
        location: {
          coordinates: scanData.location.coordinates,
          address: scanData.location.address
        },
        photos: processedPhotos,
        voiceLogs: processedVoiceLogs,
        notes: scanData.notes,
        verificationStatus: 'pending'
      };

      // Add scan to QR code
      await qrCode.addScan(scanEvent);

      // Update transport status based on QR code type
      await this.updateTransportStatus(qrCode.transportId, qrCode.type);

      // Add to blockchain if enterprise subscription
      if (await this.shouldUseBlockchain(qrCode.organizationId)) {
        await this.recordScanOnBlockchain(qrCode, scanEvent);
      }

      return {
        success: true,
        qrCodeId: qrCode.qrCodeId,
        transportId: qrCode.transportId,
        caseId: qrCode.caseId,
        type: qrCode.type,
        scanEvent,
        message: 'QR code scanned successfully'
      };
    } catch (error) {
      console.error('Error scanning QR code:', error);
      throw error;
    }
  }

  /**
   * Process and upload photos
   * @param {Array} photos - Array of photo data
   * @returns {Promise<Array>} Processed photos
   */
  async processPhotos(photos) {
    const processedPhotos = [];

    for (const photo of photos) {
      try {
        // Upload to IPFS or cloud storage
        const photoUrl = await this.uploadPhoto(photo.data);
        
        processedPhotos.push({
          photoUrl,
          photoId: `photo_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
          timestamp: new Date(),
          metadata: {
            gps: photo.gps,
            device: photo.device,
            orientation: photo.orientation
          }
        });
      } catch (error) {
        console.error('Error processing photo:', error);
      }
    }

    return processedPhotos;
  }

  /**
   * Process and upload voice logs
   * @param {Array} voiceLogs - Array of voice log data
   * @returns {Promise<Array>} Processed voice logs
   */
  async processVoiceLogs(voiceLogs) {
    const processedVoiceLogs = [];

    for (const voiceLog of voiceLogs) {
      try {
        // Upload audio to IPFS or cloud storage
        const audioUrl = await this.uploadAudio(voiceLog.data);
        
        processedVoiceLogs.push({
          audioUrl,
          transcription: voiceLog.transcription,
          duration: voiceLog.duration,
          timestamp: new Date()
        });
      } catch (error) {
        console.error('Error processing voice log:', error);
      }
    }

    return processedVoiceLogs;
  }

  /**
   * Upload photo to storage
   * @param {string} photoData - Base64 photo data
   * @returns {Promise<string>} Photo URL
   */
  async uploadPhoto(photoData) {
    try {
      // For now, return a placeholder URL
      // In production, upload to IPFS or cloud storage
      return `https://storage.example.com/photos/${Date.now()}.jpg`;
    } catch (error) {
      console.error('Error uploading photo:', error);
      throw error;
    }
  }

  /**
   * Upload audio to storage
   * @param {string} audioData - Base64 audio data
   * @returns {Promise<string>} Audio URL
   */
  async uploadAudio(audioData) {
    try {
      // For now, return a placeholder URL
      // In production, upload to IPFS or cloud storage
      return `https://storage.example.com/audio/${Date.now()}.mp3`;
    } catch (error) {
      console.error('Error uploading audio:', error);
      throw error;
    }
  }

  /**
   * Update transport status based on QR code scan
   * @param {string} transportId - Transport ID
   * @param {string} qrCodeType - Type of QR code scanned
   */
  async updateTransportStatus(transportId, qrCodeType) {
    try {
      const transport = await Transport.findById(transportId);
      if (!transport) return;

      let newStatus = transport.status;

      switch (qrCodeType) {
        case 'pickup':
          newStatus = 'in_transit';
          break;
        case 'transfer':
          newStatus = 'transferring';
          break;
        case 'delivery':
          newStatus = 'delivered';
          break;
        case 'verification':
          newStatus = 'verified';
          break;
      }

      if (newStatus !== transport.status) {
        transport.status = newStatus;
        transport.lastStatusUpdate = new Date();
        await transport.save();
      }
    } catch (error) {
      console.error('Error updating transport status:', error);
    }
  }

  /**
   * Check if organization should use blockchain
   * @param {string} organizationId - Organization ID
   * @returns {Promise<boolean>} Whether to use blockchain
   */
  async shouldUseBlockchain(organizationId) {
    try {
      // Check if organization has enterprise subscription
      const Subscription = require('../models/Subscription');
      const subscription = await Subscription.findOne({ 
        organizationId, 
        status: 'active',
        tier: 'enterprise'
      });
      
      return !!subscription;
    } catch (error) {
      console.error('Error checking blockchain requirement:', error);
      return false;
    }
  }

  /**
   * Add QR code to blockchain
   * @param {Object} qrCode - QR code document
   */
  async addToBlockchain(qrCode) {
    try {
      const blockchainData = {
        qrCodeId: qrCode.qrCodeId,
        transportId: qrCode.transportId.toString(),
        caseId: qrCode.caseId,
        type: qrCode.type,
        timestamp: qrCode.metadata.generatedAt,
        organizationId: qrCode.organizationId.toString()
      };

      const result = await blockchainService.createEvent('qr_code_generated', blockchainData);
      
      qrCode.blockchainHash = result.hash;
      qrCode.blockchainTransactionId = result.transactionId;
      await qrCode.save();
    } catch (error) {
      console.error('Error adding QR code to blockchain:', error);
    }
  }

  /**
   * Record scan event on blockchain
   * @param {Object} qrCode - QR code document
   * @param {Object} scanEvent - Scan event data
   */
  async recordScanOnBlockchain(qrCode, scanEvent) {
    try {
      const blockchainData = {
        qrCodeId: qrCode.qrCodeId,
        transportId: qrCode.transportId.toString(),
        caseId: qrCode.caseId,
        type: qrCode.type,
        scannedBy: scanEvent.scannedBy.toString(),
        scannedAt: scanEvent.scannedAt,
        location: scanEvent.location
      };

      await blockchainService.createEvent('qr_code_scanned', blockchainData);
    } catch (error) {
      console.error('Error recording scan on blockchain:', error);
    }
  }

  /**
   * Get QR codes for a transport
   * @param {string} transportId - Transport ID
   * @returns {Promise<Array>} QR codes
   */
  async getTransportQRCodes(transportId) {
    try {
      const qrCodes = await QRCodeModel.findByTransport(transportId);
      return qrCodes;
    } catch (error) {
      console.error('Error getting transport QR codes:', error);
      throw error;
    }
  }

  /**
   * Verify QR code validity
   * @param {string} qrCodeId - QR code ID
   * @returns {Promise<Object>} Verification result
   */
  async verifyQRCode(qrCodeId) {
    try {
      const qrCode = await QRCodeModel.findOne({ qrCodeId });
      if (!qrCode) {
        return { valid: false, reason: 'QR code not found' };
      }

      return qrCode.verifyQRCode();
    } catch (error) {
      console.error('Error verifying QR code:', error);
      throw error;
    }
  }

  /**
   * Get QR code statistics
   * @param {string} organizationId - Organization ID
   * @returns {Promise<Object>} Statistics
   */
  async getQRCodeStats(organizationId) {
    try {
      const stats = await QRCodeModel.aggregate([
        { $match: { organizationId: organizationId } },
        {
          $group: {
            _id: '$status',
            count: { $sum: 1 }
          }
        }
      ]);

      return stats;
    } catch (error) {
      console.error('Error getting QR code stats:', error);
      throw error;
    }
  }
}

module.exports = new QRCodeService();
