const { aiService } = require('./ai');
const { sendEmail } = require('./email');

class NotificationService {
  constructor() {
    this.isInitialized = false;
  }

  async initialize() {
    try {
      this.isInitialized = true;
      console.log('Notification service initialized successfully');
    } catch (error) {
      console.error('Failed to initialize notification service:', error);
      throw error;
    }
  }

  // Send voice notification (replaces SMS)
  async sendVoiceNotification(to, message, options = {}) {
    try {
      if (!this.isInitialized) {
        throw new Error('Notification service not initialized');
      }

      // Use Retell to make an intelligent AI voice call instead of SMS
      const callResult = await aiService.createIntelligentVoiceNotification(to, message, options.context || {});
      
      console.log(`Intelligent voice notification call initiated to ${to}: ${callResult.callId}`);
      
      return {
        success: true,
        callId: callResult.callId,
        status: callResult.status,
        type: 'intelligent_voice_notification'
      };
    } catch (error) {
      console.error('Failed to send voice notification:', error);
      throw error;
    }
  }

  // Send push notification
  async sendPushNotification(userId, notification, options = {}) {
    try {
      // This would integrate with a push notification service like Firebase
      // For now, we'll log the notification
      console.log(`Push notification for user ${userId}:`, notification);
      
      return {
        success: true,
        messageId: `push_${Date.now()}`,
        status: 'sent'
      };
    } catch (error) {
      console.error('Failed to send push notification:', error);
      throw error;
    }
  }

  // Send AI voice call (replaces Twilio voice calls)
  async sendVoiceCall(to, message, options = {}) {
    try {
      if (!this.isInitialized) {
        throw new Error('Notification service not initialized');
      }

      // Use Retell to create an intelligent AI voice call
      const callResult = await aiService.createIntelligentVoiceNotification(to, message, options.context || {});
      
      console.log(`Intelligent AI voice call initiated to ${to}: ${callResult.callId}`);
      
      return {
        success: true,
        callId: callResult.callId,
        status: callResult.status,
        type: 'intelligent_ai_voice_call'
      };
    } catch (error) {
      console.error('Failed to send voice call:', error);
      throw error;
    }
  }

  // Send multi-channel notification
  async sendMultiChannelNotification(user, notification, channels = ['email']) {
    const results = {};

    try {
      // Send email notification
      if (channels.includes('email') && user.email) {
        try {
          results.email = await sendEmail({
            to: user.email,
            subject: notification.subject,
            template: notification.template,
            data: notification.data
          });
        } catch (emailError) {
          console.error('Email notification failed:', emailError);
          results.email = { success: false, error: emailError.message };
        }
      }

      // Send voice notification (replaces SMS)
      if (channels.includes('sms') && user.phone) {
        try {
          results.voice = await this.sendVoiceNotification(user.phone, notification.voiceMessage);
        } catch (voiceError) {
          console.error('Voice notification failed:', voiceError);
          results.voice = { success: false, error: voiceError.message };
        }
      }

      // Send push notification
      if (channels.includes('push')) {
        try {
          results.push = await this.sendPushNotification(user._id, notification);
        } catch (pushError) {
          console.error('Push notification failed:', pushError);
          results.push = { success: false, error: pushError.message };
        }
      }

      // Send voice call
      if (channels.includes('voice') && user.phone) {
        try {
          results.voiceCall = await this.sendVoiceCall(user.phone, notification.voiceMessage);
        } catch (voiceCallError) {
          console.error('Voice call failed:', voiceCallError);
          results.voiceCall = { success: false, error: voiceCallError.message };
        }
      }

      return {
        success: true,
        results
      };
    } catch (error) {
      console.error('Multi-channel notification failed:', error);
      throw error;
    }
  }

  // Send emergency notification
  async sendEmergencyNotification(transport, emergencyType, details) {
    try {
      const notification = {
        subject: `Emergency Alert - Transport ${transport.caseId}`,
        template: 'emergency-alert',
        data: {
          transportId: transport._id,
          caseId: transport.caseId,
          emergencyType,
          details,
          timestamp: new Date().toISOString()
        },
        voiceMessage: `Emergency alert for transport ${transport.caseId}. ${emergencyType}: ${details}`
      };

      // Notify assigned technician
      if (transport.assignedTechnician) {
        await this.sendMultiChannelNotification(
          transport.assignedTechnician,
          notification,
          ['email', 'voice', 'push']
        );
      }

      // Notify backup technician
      if (transport.backupTechnician) {
        await this.sendMultiChannelNotification(
          transport.backupTechnician,
          notification,
          ['email', 'voice', 'push']
        );
      }

      // Notify family members
      for (const familyMember of transport.familyMembers) {
        if (familyMember.user) {
          await this.sendMultiChannelNotification(
            familyMember.user,
            notification,
            ['email', 'voice']
          );
        }
      }

      return {
        success: true,
        message: 'Emergency notifications sent successfully'
      };
    } catch (error) {
      console.error('Failed to send emergency notification:', error);
      throw error;
    }
  }

  // Send status update notification
  async sendStatusUpdateNotification(transport, status, notes = '') {
    try {
      const notification = {
        subject: `Transport Status Update - ${transport.caseId}`,
        template: 'status-update',
        data: {
          transportId: transport._id,
          caseId: transport.caseId,
          status,
          notes,
          timestamp: new Date().toISOString()
        },
        voiceMessage: `Status update for transport ${transport.caseId}. Current status: ${status}. ${notes}`
      };

      // Notify family members
      for (const familyMember of transport.familyMembers) {
        if (familyMember.user) {
          await this.sendMultiChannelNotification(
            familyMember.user,
            notification,
            ['email', 'voice']
          );
        }
      }

      return {
        success: true,
        message: 'Status update notifications sent successfully'
      };
    } catch (error) {
      console.error('Failed to send status update notification:', error);
      throw error;
    }
  }

  // Send reminder notification
  async sendReminderNotification(user, reminder) {
    try {
      const notification = {
        subject: reminder.subject,
        template: 'reminder',
        data: {
          reminderId: reminder._id,
          title: reminder.title,
          description: reminder.description,
          dueDate: reminder.dueDate,
          priority: reminder.priority
        },
        voiceMessage: `Reminder: ${reminder.title}. ${reminder.description}`
      };

      await this.sendMultiChannelNotification(
        user,
        notification,
        ['email', 'voice', 'push']
      );

      return {
        success: true,
        message: 'Reminder notification sent successfully'
      };
    } catch (error) {
      console.error('Failed to send reminder notification:', error);
      throw error;
    }
  }

  // Send compliance alert
  async sendComplianceAlert(user, alert) {
    try {
      const notification = {
        subject: `Compliance Alert - ${alert.type}`,
        template: 'compliance-alert',
        data: {
          alertId: alert._id,
          type: alert.type,
          severity: alert.severity,
          description: alert.description,
          actionRequired: alert.actionRequired,
          dueDate: alert.dueDate
        },
        voiceMessage: `Compliance alert: ${alert.type}. ${alert.description}. Action required: ${alert.actionRequired}`
      };

      await this.sendMultiChannelNotification(
        user,
        notification,
        ['email', 'voice', 'push']
      );

      return {
        success: true,
        message: 'Compliance alert sent successfully'
      };
    } catch (error) {
      console.error('Failed to send compliance alert:', error);
      throw error;
    }
  }

  // Send welcome notification
  async sendWelcomeNotification(user) {
    try {
      const notification = {
        subject: 'Welcome to Obitix',
        template: 'welcome',
        data: {
          firstName: user.firstName,
          lastName: user.lastName,
          role: user.role,
          organizationName: user.organization?.name || 'Obitix'
        },
        voiceMessage: `Welcome to Obitix, ${user.firstName}. Your account has been successfully created.`
      };

      await this.sendMultiChannelNotification(
        user,
        notification,
        ['email', 'voice']
      );

      return {
        success: true,
        message: 'Welcome notification sent successfully'
      };
    } catch (error) {
      console.error('Failed to send welcome notification:', error);
      throw error;
    }
  }

  // Send password reset notification
  async sendPasswordResetNotification(user, resetToken) {
    try {
      const resetUrl = `${process.env.FRONTEND_URL}/reset-password?token=${resetToken}`;
      
      const notification = {
        subject: 'Password Reset Request',
        template: 'password-reset',
        data: {
          firstName: user.firstName,
          resetUrl,
          expiresIn: '1 hour'
        },
        voiceMessage: `Password reset requested for your Obitix account. Please check your email for reset instructions.`
      };

      await this.sendMultiChannelNotification(
        user,
        notification,
        ['email', 'voice']
      );

      return {
        success: true,
        message: 'Password reset notification sent successfully'
      };
    } catch (error) {
      console.error('Failed to send password reset notification:', error);
      throw error;
    }
  }
}

// Create singleton instance
const notificationService = new NotificationService();

// Export functions for direct use
const sendVoiceNotification = (to, message, options) => 
  notificationService.sendVoiceNotification(to, message, options);
const sendPushNotification = (userId, notification, options) => 
  notificationService.sendPushNotification(userId, notification, options);
const sendVoiceCall = (to, message, options) => 
  notificationService.sendVoiceCall(to, message, options);
const sendMultiChannelNotification = (user, notification, channels) =>
  notificationService.sendMultiChannelNotification(user, notification, channels);
const sendEmergencyNotification = (transport, emergencyType, details) =>
  notificationService.sendEmergencyNotification(transport, emergencyType, details);
const sendStatusUpdateNotification = (transport, status, notes) =>
  notificationService.sendStatusUpdateNotification(transport, status, notes);
const sendReminderNotification = (user, reminder) =>
  notificationService.sendReminderNotification(user, reminder);
const sendComplianceAlert = (user, alert) =>
  notificationService.sendComplianceAlert(user, alert);
const sendWelcomeNotification = (user) =>
  notificationService.sendWelcomeNotification(user);
const sendPasswordResetNotification = (user, resetToken) =>
  notificationService.sendPasswordResetNotification(user, resetToken);

module.exports = {
  notificationService,
  sendVoiceNotification,
  sendPushNotification,
  sendVoiceCall,
  sendMultiChannelNotification,
  sendEmergencyNotification,
  sendStatusUpdateNotification,
  sendReminderNotification,
  sendComplianceAlert,
  sendWelcomeNotification,
  sendPasswordResetNotification
};
