const express = require('express');
const router = express.Router();
const { verifyToken, authorize, requireOrganizationAccess } = require('../middleware/auth');
const { rateLimiter } = require('../middleware/rateLimiter');
const Organization = require('../models/Organization');
const User = require('../models/User');
const Transport = require('../models/Transport');

// Apply rate limiting to all organization routes
router.use(rateLimiter);

/**
 * @route POST /api/organizations
 * @desc Create a new organization (funeral home)
 * @access Private (Admin only)
 */
router.post('/', verifyToken, authorize(['admin']), async (req, res) => {
  try {
    const {
      name,
      type,
      contactInfo,
      address,
      businessDetails,
      subscription,
      settings
    } = req.body;

    if (!name || !contactInfo || !address) {
      return res.status(400).json({
        success: false,
        message: 'Name, contact info, and address are required'
      });
    }

    // Check if organization already exists
    const existingOrg = await Organization.findOne({ 
      'contactInfo.email': contactInfo.email 
    });

    if (existingOrg) {
      return res.status(400).json({
        success: false,
        message: 'Organization with this email already exists'
      });
    }

    const organization = new Organization({
      name,
      type: type || 'funeral_home',
      contactInfo,
      address,
      businessDetails,
      subscription: {
        plan: subscription?.plan || 'basic',
        status: subscription?.status || 'active',
        startDate: new Date(),
        ...subscription
      },
      settings: {
        notifications: {
          transportUpdates: true,
          emergencyAlerts: true,
          custodyEvents: true,
          familyUpdates: true,
          reminders: true
        },
        features: {
          blockchain: true,
          aiVoice: true,
          gpsTracking: true,
          qrScanning: true,
          voiceLogging: true
        },
        branding: {
          logo: null,
          colors: {
            primary: '#1a365d',
            secondary: '#2d3748',
            accent: '#3182ce'
          }
        },
        compliance: {
          gdpr: true,
          hipaa: true,
          auditTrails: true,
          dataRetention: 7 // years
        },
        ...settings
      }
    });

    await organization.save();

    res.status(201).json({
      success: true,
      message: 'Organization created successfully',
      organization
    });
  } catch (error) {
    console.error('Error creating organization:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to create organization'
    });
  }
});

/**
 * @route GET /api/organizations
 * @desc Get all organizations (admin) or user's organization
 * @access Private
 */
router.get('/', verifyToken, async (req, res) => {
  try {
    const { page = 1, limit = 20, search, status } = req.query;
    const user = req.user;

    // Admin can see all organizations
    if (user.role === 'admin') {
      const query = {};
      
      if (search) {
        query.$or = [
          { name: { $regex: search, $options: 'i' } },
          { 'contactInfo.email': { $regex: search, $options: 'i' } },
          { 'address.city': { $regex: search, $options: 'i' } }
        ];
      }

      if (status) {
        query['subscription.status'] = status;
      }

      const organizations = await Organization.find(query)
        .populate('staff', 'firstName lastName email role')
        .populate('equipment', 'name type status')
        .sort({ createdAt: -1 })
        .limit(limit * 1)
        .skip((page - 1) * limit)
        .exec();

      const total = await Organization.countDocuments(query);

      res.json({
        success: true,
        organizations,
        pagination: {
          currentPage: parseInt(page),
          totalPages: Math.ceil(total / limit),
          totalOrganizations: total,
          hasNextPage: page * limit < total,
          hasPrevPage: page > 1
        }
      });
    } else {
      // Regular users can only see their organization
      if (!user.organizationId) {
        return res.status(404).json({
          success: false,
          message: 'User not associated with any organization'
        });
      }

      const organization = await Organization.findById(user.organizationId)
        .populate('staff', 'firstName lastName email role status')
        .populate('equipment', 'name type status lastMaintenance')
        .populate('partners', 'name type contactInfo');

      if (!organization) {
        return res.status(404).json({
          success: false,
          message: 'Organization not found'
        });
      }

      res.json({
        success: true,
        organization
      });
    }
  } catch (error) {
    console.error('Error fetching organizations:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch organizations'
    });
  }
});

/**
 * @route GET /api/organizations/:id
 * @desc Get organization by ID
 * @access Private (Admin or organization member)
 */
router.get('/:id', verifyToken, requireOrganizationAccess, async (req, res) => {
  try {
    const organization = await Organization.findById(req.params.id)
      .populate('staff', 'firstName lastName email role status lastLogin')
      .populate('equipment', 'name type status lastMaintenance nextMaintenance')
      .populate('partners', 'name type contactInfo address')
      .populate('statistics.transports', 'transportId status createdAt');

    if (!organization) {
      return res.status(404).json({
        success: false,
        message: 'Organization not found'
      });
    }

    res.json({
      success: true,
      organization
    });
  } catch (error) {
    console.error('Error fetching organization:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch organization'
    });
  }
});

/**
 * @route PUT /api/organizations/:id
 * @desc Update organization
 * @access Private (Admin or organization admin)
 */
router.put('/:id', verifyToken, requireOrganizationAccess, async (req, res) => {
  try {
    const { id } = req.params;
    const user = req.user;
    const updateData = req.body;

    // Check if user has permission to update this organization
    if (user.role !== 'admin' && user.organizationId?.toString() !== id) {
      return res.status(403).json({
        success: false,
        message: 'Access denied'
      });
    }

    // Remove sensitive fields that shouldn't be updated directly
    delete updateData._id;
    delete updateData.createdAt;
    delete updateData.updatedAt;

    const organization = await Organization.findByIdAndUpdate(
      id,
      { $set: updateData },
      { new: true, runValidators: true }
    );

    if (!organization) {
      return res.status(404).json({
        success: false,
        message: 'Organization not found'
      });
    }

    res.json({
      success: true,
      message: 'Organization updated successfully',
      organization
    });
  } catch (error) {
    console.error('Error updating organization:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update organization'
    });
  }
});

/**
 * @route DELETE /api/organizations/:id
 * @desc Delete organization (soft delete)
 * @access Private (Admin only)
 */
router.delete('/:id', verifyToken, authorize(['admin']), async (req, res) => {
  try {
    const { id } = req.params;

    const organization = await Organization.findByIdAndUpdate(
      id,
      { 
        $set: { 
          'subscription.status': 'inactive',
          'settings.deleted': true,
          deletedAt: new Date()
        }
      },
      { new: true }
    );

    if (!organization) {
      return res.status(404).json({
        success: false,
        message: 'Organization not found'
      });
    }

    // Deactivate all staff members
    await User.updateMany(
      { organizationId: id },
      { $set: { status: 'inactive', deactivatedAt: new Date() } }
    );

    res.json({
      success: true,
      message: 'Organization deactivated successfully'
    });
  } catch (error) {
    console.error('Error deactivating organization:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to deactivate organization'
    });
  }
});

/**
 * @route POST /api/organizations/:id/staff
 * @desc Add staff member to organization
 * @access Private (Admin or organization admin)
 */
router.post('/:id/staff', verifyToken, requireOrganizationAccess, async (req, res) => {
  try {
    const { id } = req.params;
    const { userId, role, permissions } = req.body;

    if (!userId || !role) {
      return res.status(400).json({
        success: false,
        message: 'User ID and role are required'
      });
    }

    // Check if user exists
    const user = await User.findById(userId);
    if (!user) {
      return res.status(404).json({
        success: false,
        message: 'User not found'
      });
    }

    // Check if user is already in an organization
    if (user.organizationId && user.organizationId.toString() !== id) {
      return res.status(400).json({
        success: false,
        message: 'User is already associated with another organization'
      });
    }

    // Update user's organization and role
    await User.findByIdAndUpdate(userId, {
      $set: {
        organizationId: id,
        role,
        permissions: permissions || [],
        status: 'active'
      }
    });

    // Add user to organization's staff list
    await Organization.findByIdAndUpdate(id, {
      $addToSet: { staff: userId }
    });

    res.json({
      success: true,
      message: 'Staff member added successfully'
    });
  } catch (error) {
    console.error('Error adding staff member:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to add staff member'
    });
  }
});

/**
 * @route DELETE /api/organizations/:id/staff/:userId
 * @desc Remove staff member from organization
 * @access Private (Admin or organization admin)
 */
router.delete('/:id/staff/:userId', verifyToken, requireOrganizationAccess, async (req, res) => {
  try {
    const { id, userId } = req.params;

    // Remove user from organization
    await User.findByIdAndUpdate(userId, {
      $set: {
        organizationId: null,
        status: 'inactive',
        deactivatedAt: new Date()
      }
    });

    // Remove user from organization's staff list
    await Organization.findByIdAndUpdate(id, {
      $pull: { staff: userId }
    });

    res.json({
      success: true,
      message: 'Staff member removed successfully'
    });
  } catch (error) {
    console.error('Error removing staff member:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to remove staff member'
    });
  }
});

/**
 * @route GET /api/organizations/:id/statistics
 * @desc Get organization statistics
 * @access Private (Admin or organization member)
 */
router.get('/:id/statistics', verifyToken, requireOrganizationAccess, async (req, res) => {
  try {
    const { id } = req.params;
    const { period = '30d' } = req.query;

    // Calculate date range
    const now = new Date();
    let startDate;
    
    switch (period) {
      case '7d':
        startDate = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000);
        break;
      case '30d':
        startDate = new Date(now.getTime() - 30 * 24 * 60 * 60 * 1000);
        break;
      case '90d':
        startDate = new Date(now.getTime() - 90 * 24 * 60 * 60 * 1000);
        break;
      case '1y':
        startDate = new Date(now.getTime() - 365 * 24 * 60 * 60 * 1000);
        break;
      default:
        startDate = new Date(now.getTime() - 30 * 24 * 60 * 60 * 1000);
    }

    // Get transport statistics
    const transportStats = await Transport.aggregate([
      { $match: { organizationId: id, createdAt: { $gte: startDate } } },
      {
        $group: {
          _id: '$status',
          count: { $sum: 1 }
        }
      }
    ]);

    // Get daily transport counts
    const dailyTransports = await Transport.aggregate([
      { $match: { organizationId: id, createdAt: { $gte: startDate } } },
      {
        $group: {
          _id: {
            $dateToString: { format: '%Y-%m-%d', date: '$createdAt' }
          },
          count: { $sum: 1 }
        }
      },
      { $sort: { '_id': 1 } }
    ]);

    // Get staff statistics
    const staffStats = await User.aggregate([
      { $match: { organizationId: id } },
      {
        $group: {
          _id: '$role',
          count: { $sum: 1 },
          active: {
            $sum: { $cond: [{ $eq: ['$status', 'active'] }, 1, 0] }
          }
        }
      }
    ]);

    // Get equipment statistics
    const organization = await Organization.findById(id).populate('equipment');
    const equipmentStats = {
      total: organization.equipment.length,
      active: organization.equipment.filter(e => e.status === 'active').length,
      maintenance: organization.equipment.filter(e => e.status === 'maintenance').length,
      inactive: organization.equipment.filter(e => e.status === 'inactive').length
    };

    res.json({
      success: true,
      statistics: {
        period,
        startDate,
        endDate: now,
        transports: {
          total: transportStats.reduce((sum, stat) => sum + stat.count, 0),
          byStatus: transportStats,
          daily: dailyTransports
        },
        staff: {
          total: staffStats.reduce((sum, stat) => sum + stat.count, 0),
          byRole: staffStats,
          active: staffStats.reduce((sum, stat) => sum + stat.active, 0)
        },
        equipment: equipmentStats
      }
    });
  } catch (error) {
    console.error('Error fetching organization statistics:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch organization statistics'
    });
  }
});

/**
 * @route POST /api/organizations/:id/equipment
 * @desc Add equipment to organization
 * @access Private (Admin or organization admin)
 */
router.post('/:id/equipment', verifyToken, requireOrganizationAccess, async (req, res) => {
  try {
    const { id } = req.params;
    const { name, type, serialNumber, purchaseDate, warrantyExpiry, maintenanceSchedule } = req.body;

    if (!name || !type) {
      return res.status(400).json({
        success: false,
        message: 'Name and type are required'
      });
    }

    const equipment = {
      id: require('crypto').randomUUID(),
      name,
      type,
      serialNumber,
      status: 'active',
      purchaseDate: purchaseDate ? new Date(purchaseDate) : new Date(),
      warrantyExpiry: warrantyExpiry ? new Date(warrantyExpiry) : null,
      maintenanceSchedule: maintenanceSchedule || {
        interval: 90, // days
        lastMaintenance: null,
        nextMaintenance: null
      },
      createdAt: new Date()
    };

    await Organization.findByIdAndUpdate(id, {
      $push: { equipment }
    });

    res.json({
      success: true,
      message: 'Equipment added successfully',
      equipment
    });
  } catch (error) {
    console.error('Error adding equipment:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to add equipment'
    });
  }
});

/**
 * @route PUT /api/organizations/:id/equipment/:equipmentId
 * @desc Update equipment
 * @access Private (Admin or organization admin)
 */
router.put('/:id/equipment/:equipmentId', verifyToken, requireOrganizationAccess, async (req, res) => {
  try {
    const { id, equipmentId } = req.params;
    const updateData = req.body;

    const organization = await Organization.findById(id);
    if (!organization) {
      return res.status(404).json({
        success: false,
        message: 'Organization not found'
      });
    }

    const equipmentIndex = organization.equipment.findIndex(e => e.id === equipmentId);
    if (equipmentIndex === -1) {
      return res.status(404).json({
        success: false,
        message: 'Equipment not found'
      });
    }

    // Update equipment
    organization.equipment[equipmentIndex] = {
      ...organization.equipment[equipmentIndex],
      ...updateData,
      updatedAt: new Date()
    };

    await organization.save();

    res.json({
      success: true,
      message: 'Equipment updated successfully',
      equipment: organization.equipment[equipmentIndex]
    });
  } catch (error) {
    console.error('Error updating equipment:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to update equipment'
    });
  }
});

/**
 * @route DELETE /api/organizations/:id/equipment/:equipmentId
 * @desc Remove equipment from organization
 * @access Private (Admin or organization admin)
 */
router.delete('/:id/equipment/:equipmentId', verifyToken, requireOrganizationAccess, async (req, res) => {
  try {
    const { id, equipmentId } = req.params;

    await Organization.findByIdAndUpdate(id, {
      $pull: { equipment: { id: equipmentId } }
    });

    res.json({
      success: true,
      message: 'Equipment removed successfully'
    });
  } catch (error) {
    console.error('Error removing equipment:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to remove equipment'
    });
  }
});

/**
 * @route GET /api/organizations/:id/audit-logs
 * @desc Get organization audit logs
 * @access Private (Admin or organization admin)
 */
router.get('/:id/audit-logs', verifyToken, requireOrganizationAccess, async (req, res) => {
  try {
    const { id } = req.params;
    const { page = 1, limit = 50, action, userId, startDate, endDate } = req.query;

    const query = { organizationId: id };

    if (action) query.action = action;
    if (userId) query.userId = userId;
    if (startDate || endDate) {
      query.timestamp = {};
      if (startDate) query.timestamp.$gte = new Date(startDate);
      if (endDate) query.timestamp.$lte = new Date(endDate);
    }

    // This would typically query an audit log collection
    // For now, we'll return a placeholder
    const auditLogs = [];
    const total = 0;

    res.json({
      success: true,
      auditLogs,
      pagination: {
        currentPage: parseInt(page),
        totalPages: Math.ceil(total / limit),
        totalLogs: total,
        hasNextPage: page * limit < total,
        hasPrevPage: page > 1
      }
    });
  } catch (error) {
    console.error('Error fetching audit logs:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to fetch audit logs'
    });
  }
});

module.exports = router;
