const express = require('express');
const router = express.Router();
const multer = require('multer');
const path = require('path');
const fs = require('fs');
const { v4: uuidv4 } = require('uuid');
const OpenAI = require('openai');
const axios = require('axios');

// Helper function to get transport context
const getTransportContext = async (transportId) => {
  try {
    const Transport = require('../models/Transport');
    const transport = await Transport.findById(transportId);
    if (transport) {
      return `
        TRANSPORT CONTEXT:
        Case ID: ${transport.caseId}
        Deceased: ${transport.deceasedName}
        Status: ${transport.status}
        Technician: ${transport.technicianName}
        Pickup: ${transport.pickupLocation}
        Delivery: ${transport.deliveryLocation}
        Created: ${transport.createdAt}
        Updated: ${transport.updatedAt}
        
        IMPORTANT: You have access to this transport information. When family members ask about their transport, provide specific details about their case. You can answer questions about:
        - The deceased person's name
        - Current transport status
        - Technician assigned to the transport
        - Pickup and delivery locations
        - Any other transport-related information
        
        Be compassionate and helpful while providing accurate information from the transport data.
      `;
    }
  } catch (error) {
    console.log('Error fetching transport context:', error.message);
  }
  return '';
};
const { verifyToken } = require('../middleware/auth');
const { checkFeatureAccess } = require('../middleware/subscription');
const { uploadToIPFS } = require('../services/ipfs');

// Initialize OpenAI
const openai = new OpenAI({
  apiKey: process.env.OPENAI_API_KEY,
});

// Configure multer for audio uploads
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    const uploadDir = 'uploads/audio';
    if (!fs.existsSync(uploadDir)) {
      fs.mkdirSync(uploadDir, { recursive: true });
    }
    cb(null, uploadDir);
  },
  filename: (req, file, cb) => {
    const uniqueName = `${uuidv4()}-${Date.now()}${path.extname(file.originalname)}`;
    cb(null, uniqueName);
  }
});

const upload = multer({
  storage: storage,
  limits: {
    fileSize: 25 * 1024 * 1024, // 25MB limit for audio files
  },
  fileFilter: (req, file, cb) => {
    const allowedTypes = /mp3|wav|m4a|aac|ogg/;
    const extname = allowedTypes.test(path.extname(file.originalname).toLowerCase());
    const mimetype = allowedTypes.test(file.mimetype);

    if (mimetype && extname) {
      return cb(null, true);
    } else {
      cb(new Error('Only audio files are allowed'));
    }
  }
});

// Test AI service status
router.get('/status', verifyToken, async (req, res) => {
  try {
    console.log('AI Status Check - User:', req.user.email);
    
    // Check if OpenAI API key is configured
    const hasApiKey = !!process.env.OPENAI_API_KEY;
    console.log('AI Status - Has API Key:', hasApiKey);
    
    // Test OpenAI connection
    let openaiStatus = 'not_configured';
    if (hasApiKey) {
      try {
        await openai.models.list();
        openaiStatus = 'connected';
        console.log('AI Status - OpenAI connection successful');
      } catch (openaiError) {
        openaiStatus = 'error';
        console.error('AI Status - OpenAI connection failed:', openaiError.message);
      }
    }
    
    res.json({
      success: true,
      status: {
        openai: openaiStatus,
        hasApiKey,
        timestamp: new Date().toISOString()
      }
    });
  } catch (error) {
    console.error('AI Status Check Error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to check AI service status',
      error: error.message
    });
  }
});

// Test voice chat without OpenAI (for debugging)
router.post('/test-voice-simple', verifyToken, checkFeatureAccess('voiceLoggingAssistant'), upload.single('test'), async (req, res) => {
  try {
    console.log('Test Voice Simple - Request received');
    console.log('Request body:', req.body);
    console.log('Request file:', req.file);
    
    const response = {
      success: true,
      conversation: {
        userMessage: "Test voice message",
        aiResponse: "Hello! This is a test response from your AI assistant. The voice chat system is working correctly.",
        audioResponse: 'TTS_UNAVAILABLE_' + Date.now()
      },
      formDataReceived: !!req.file,
      bodyReceived: Object.keys(req.body).length > 0
    };

    console.log('Test Voice Simple - Sending test response');
    res.json(response);
    
  } catch (error) {
    console.error('Test Voice Simple - Error:', error);
    res.status(500).json({
      success: false,
      message: 'Test failed',
      error: error.message
    });
  }
});

// Live voice call session management
router.post('/live-voice-call', verifyToken, checkFeatureAccess('voiceLoggingAssistant'), async (req, res) => {
  try {
    console.log('Live Voice Call - Creating session');
    
    const { transportId, context } = req.body;
    const userId = req.user.id;
    
    // Create a unique session ID
    const sessionId = `live_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
    
    // Store session info (in production, use Redis or database)
    const session = {
      sessionId,
      userId,
      transportId,
      context,
      createdAt: new Date(),
      isActive: true
    };
    
    console.log('Live Voice Call - Session created:', sessionId);
    
    res.json({
      success: true,
      session: session,
      message: 'Live voice call session created successfully'
    });
    
  } catch (error) {
    console.error('Live Voice Call - Error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to create live voice call session',
      error: error.message
    });
  }
});

router.post('/end-live-call', verifyToken, checkFeatureAccess('voiceLoggingAssistant'), async (req, res) => {
  try {
    console.log('Live Voice Call - Ending session');
    
    const { sessionId } = req.body;
    
    // In production, mark session as inactive in database/Redis
    console.log('Live Voice Call - Session ended:', sessionId);
    
    res.json({
      success: true,
      message: 'Live voice call session ended successfully'
    });
    
  } catch (error) {
    console.error('Live Voice Call - Error ending session:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to end live voice call session',
      error: error.message
    });
  }
});

router.post('/process-live-speech', verifyToken, checkFeatureAccess('voiceLoggingAssistant'), upload.single('audio'), async (req, res) => {
  try {
    console.log('Live Voice Call - Processing speech');
    console.log('Request body:', req.body);
    console.log('Request file:', req.file);
    
    if (!req.file) {
      console.log('No file in request');
      return res.status(400).json({
        success: false,
        message: 'No audio file provided'
      });
    }

    const { sessionId, transportId, context } = req.body;
    const userId = req.user.id;
    
    console.log('Live Voice Call - Processing speech for session:', sessionId);

    // Transcribe the speech
    let userTranscription = '';
    try {
      const transcription = await openai.audio.transcriptions.create({
        file: fs.createReadStream(req.file.path),
        model: 'whisper-1',
        language: 'en'
      });
      userTranscription = transcription.text;
      console.log('Live Voice Call - Transcription:', userTranscription);
    } catch (transcriptionError) {
      console.error('Live Voice Call - Transcription failed:', transcriptionError);
      userTranscription = 'I received your voice message but couldn\'t transcribe it. Please try again.';
    }

    // Generate AI response
    let aiResponse = '';
    try {
      console.log('Live Voice Call - Transport ID received:', transportId);
      const transportContext = transportId ? await getTransportContext(transportId) : '';
      console.log('Live Voice Call - Transport context:', transportContext ? 'Available' : 'Not available');
      
      const systemPrompt = `You are an AI assistant for Obitix, a funeral transport management system. You are helping a family member during a difficult time. Provide compassionate, empathetic support. Be sensitive, understanding, and patient. Offer comfort while providing practical information. ${transportContext} Keep responses concise, conversational, and helpful. For family members, always be compassionate and understanding.`;
      
      console.log('Live Voice Call - System prompt length:', systemPrompt.length);
      
      const completion = await openai.chat.completions.create({
        model: 'gpt-4o-mini',
        messages: [
          {
            role: 'system',
            content: systemPrompt
          },
          {
            role: 'user',
            content: userTranscription
          }
        ],
        max_tokens: 200,
        temperature: 0.7
      });
      
      aiResponse = completion.choices[0].message.content;
      console.log('Live Voice Call - AI Response:', aiResponse);
    } catch (aiError) {
      console.error('Live Voice Call - AI response failed:', aiError);
      aiResponse = 'I\'m here to help you. Please let me know how I can support you today.';
    }

    // Generate TTS for AI response
    let audioResponse = '';
    try {
      const ttsResponse = await openai.audio.speech.create({
        model: 'tts-1',
        voice: 'alloy',
        input: aiResponse,
        response_format: 'mp3'
      });

      const audioBuffer = Buffer.from(await ttsResponse.arrayBuffer());
      audioResponse = audioBuffer.toString('base64');
      console.log('Live Voice Call - TTS generated successfully');
    } catch (ttsError) {
      console.error('Live Voice Call - TTS failed:', ttsError);
      // Return text response if TTS fails
      audioResponse = 'TTS_UNAVAILABLE_' + Date.now();
    }

    // Clean up the uploaded file
    try {
      fs.unlinkSync(req.file.path);
    } catch (cleanupError) {
      console.log('Live Voice Call - File cleanup failed:', cleanupError);
    }

    res.json({
      success: true,
      userTranscription,
      aiResponse,
      audioResponse,
      sessionId
    });
    
  } catch (error) {
    console.error('Live Voice Call - Processing error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to process live speech',
      error: error.message
    });
  }
});

// Retell.ai real-time voice call endpoint
router.post('/retell-voice-call', verifyToken, checkFeatureAccess('voiceLoggingAssistant'), async (req, res) => {
  try {
    console.log('Retell Voice Call - Request received');
    
    const { transportId, context } = req.body;
    
    // Check if Retell.ai is configured
    if (!process.env.RETELL_API_KEY || !process.env.RETELL_AGENT_ID) {
      return res.status(400).json({
        success: false,
        message: 'Retell.ai not configured. Please set RETELL_API_KEY and RETELL_AGENT_ID environment variables.'
      });
    }
    
    // Get transport context for the AI
    let transportContext = '';
    if (transportId) {
      try {
        const Transport = require('../models/Transport');
        const transport = await Transport.findById(transportId);
        if (transport) {
          transportContext = `
            Transport Details:
            - Case ID: ${transport.caseId}
            - Deceased: ${transport.deceasedName}
            - Status: ${transport.status}
            - Technician: ${transport.technicianName}
            - Pickup: ${transport.pickupLocation}
            - Delivery: ${transport.deliveryLocation}
          `;
        }
      } catch (transportError) {
        console.log('Retell Voice Call - Error fetching transport:', transportError.message);
      }
    }
    
    // Try Retell.ai first, fallback to OpenAI if it fails
    try {
      // Create Retell.ai call session
      const retellResponse = await axios.post('https://api.retellai.com/v1/call', {
        agent_id: process.env.RETELL_AGENT_ID,
        audio_websocket_protocol: "web",
        audio_encoding: "s16le",
        sample_rate: 24000,
        enable_backchannel: true,
        metadata: {
          transportId: transportId || '',
          context: context || 'family-support',
          transportContext: transportContext
        }
      }, {
        headers: {
          'Authorization': `Bearer ${process.env.RETELL_API_KEY}`,
          'Content-Type': 'application/json'
        }
      });
      
      console.log('Retell Voice Call - Call session created:', retellResponse.data);
      
      res.json({
        success: true,
        callId: retellResponse.data.call_id,
        websocketUrl: retellResponse.data.websocket_url,
        message: 'Real-time voice call session created successfully',
        provider: 'retell'
      });
      
    } catch (retellError) {
      console.log('Retell.ai call creation failed, falling back to OpenAI:', retellError.message);
      
      // Fallback: Create a session ID for OpenAI-based voice chat
      const sessionId = `openai_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
      
      res.json({
        success: true,
        callId: sessionId,
        websocketUrl: null, // No WebSocket for OpenAI fallback
        message: 'Live voice chat session created using OpenAI (Retell.ai unavailable)',
        provider: 'openai',
        fallback: true
      });
    }
    
  } catch (error) {
    console.error('Retell Voice Call - Error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to create voice call session',
      error: error.message
    });
  }
});

// Test voice generation endpoint
router.post('/test-voice', verifyToken, checkFeatureAccess('voiceLoggingAssistant'), async (req, res) => {
  try {
    console.log('Test Voice - Request received');
    
    const { text = 'Hello! This is a test voice response from the AI assistant.' } = req.body;
    
    console.log('Test Voice - Generating speech for text:', text);
    
    let base64Audio = null;
    let audioLength = 0;
    let ttsStatus = 'success';
    
    try {
      // Generate speech
      const speech = await openai.audio.speech.create({
        model: 'tts-1',
        voice: 'alloy',
        input: text
      });

      console.log('Test Voice - Speech generated successfully');

      // Convert speech buffer to base64
      const buffer = Buffer.from(await speech.arrayBuffer());
      base64Audio = buffer.toString('base64');
      audioLength = base64Audio.length;
      
      console.log('Test Voice - Audio converted to base64, length:', audioLength);
      
    } catch (ttsError) {
      console.log('Test Voice - TTS failed:', ttsError.message);
      ttsStatus = 'quota_exceeded';
      
      // Create a placeholder for testing
      base64Audio = 'TTS_UNAVAILABLE_' + Date.now();
      audioLength = base64Audio.length;
      
      console.log('Test Voice - Using fallback audio placeholder');
    }

    res.json({
      success: true,
      audioData: base64Audio,
      text: text,
      audioLength: audioLength,
      ttsStatus: ttsStatus,
      message: ttsStatus === 'success' ? 'Voice generated successfully' : 'TTS quota exceeded - using placeholder'
    });
    
  } catch (error) {
    console.error('Test Voice - Error:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to generate test voice',
      error: error.message
    });
  }
});

// Transcribe audio to text
router.post('/transcribe', verifyToken, checkFeatureAccess('voiceLoggingAssistant'), upload.single('audio'), async (req, res) => {
  try {
    const file = req.file;
    const { language = 'en' } = req.body;

    if (!file) {
      return res.status(400).json({
        success: false,
        message: 'No audio file uploaded'
      });
    }

    // Transcribe audio using OpenAI Whisper
    const transcription = await openai.audio.transcriptions.create({
      file: fs.createReadStream(file.path),
      model: 'whisper-1',
      language: language,
      response_format: 'verbose_json',
      timestamp_granularities: ['word']
    });

    // Upload audio file to IPFS for storage
    let ipfsHash = null;
    try {
      const ipfsResult = await uploadToIPFS(file.path, file.filename);
      ipfsHash = ipfsResult.hash;
    } catch (ipfsError) {
      console.error('IPFS upload failed:', ipfsError);
    }

    // Clean up local file
    fs.unlinkSync(file.path);

    res.json({
      success: true,
      transcription: {
        text: transcription.text,
        language: transcription.language,
        duration: transcription.duration,
        segments: transcription.segments,
        ipfsHash
      }
    });
  } catch (error) {
    console.error('Error transcribing audio:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to transcribe audio'
    });
  }
});

// Generate AI response
router.post('/chat', verifyToken, async (req, res) => {
  try {
    console.log('AI Chat Request - User:', req.user.email, 'Role:', req.user.role);
    console.log('AI Chat Request - Body:', req.body);
    
    const { message, context, transportId, messageType = 'general', transportData } = req.body;

    if (!message) {
      console.log('AI Chat Error - No message provided');
      return res.status(400).json({
        success: false,
        message: 'Message is required'
      });
    }

    // Check if OpenAI is properly configured
    if (!process.env.OPENAI_API_KEY) {
      console.error('AI Chat Error - OpenAI API key not configured');
      return res.status(500).json({
        success: false,
        message: 'AI service not configured'
      });
    }

    // Build system prompt based on context and user role
    let systemPrompt = 'You are an AI assistant for Obitix, a funeral transport management system. ';
    
    if (req.user.role === 'technician') {
      systemPrompt += 'You are helping a transport technician. Provide professional, compassionate assistance with transport-related tasks, safety protocols, and documentation. ';
    } else if (req.user.role === 'family') {
      systemPrompt += 'You are helping a family member during a difficult time. Provide compassionate, empathetic support. Be sensitive, understanding, and patient. Offer comfort while providing practical information. ';
    } else if (req.user.role === 'staff' || req.user.role === 'admin') {
      systemPrompt += 'You are helping funeral home staff. Provide professional assistance with administrative tasks, compliance, and customer service. ';
    }

    // Add transport context for family members
    if (req.user.role === 'family' && transportData) {
      systemPrompt += `\n\nTRANSPORT CONTEXT:\n`;
      
      if (transportData.transports && transportData.transports.length > 0) {
        systemPrompt += `The family member has ${transportData.transports.length} transport(s):\n`;
        transportData.transports.forEach((transport, index) => {
          systemPrompt += `${index + 1}. Case ID: ${transport.caseId}\n`;
          systemPrompt += `   Deceased: ${transport.deceased.firstName} ${transport.deceased.lastName}\n`;
          systemPrompt += `   Status: ${transport.status}\n`;
          systemPrompt += `   Priority: ${transport.priority}\n`;
          if (transport.assignedTechnician) {
            systemPrompt += `   Technician: ${transport.assignedTechnician.firstName} ${transport.assignedTechnician.lastName}\n`;
          }
          if (transport.scheduledPickupTime) {
            systemPrompt += `   Scheduled Pickup: ${new Date(transport.scheduledPickupTime).toLocaleString()}\n`;
          }
          systemPrompt += `   Pickup: ${transport.pickupLocation.name || transport.pickupLocation.address?.street || 'Location'}\n`;
          systemPrompt += `   Delivery: ${transport.deliveryLocation.name || transport.deliveryLocation.address?.street || 'Location'}\n\n`;
        });
      } else {
        systemPrompt += `The family member currently has no active transports.\n\n`;
      }

      if (transportData.transportSummary) {
        systemPrompt += `TRANSPORT SUMMARY: ${transportData.transportSummary}\n\n`;
      }
    }

    if (transportId) {
      systemPrompt += `This conversation is related to transport ID: ${transportId}. `;
    }

    if (messageType === 'family-support') {
      systemPrompt += `\n\nSPECIFIC INSTRUCTIONS FOR FAMILY SUPPORT:\n`;
      systemPrompt += `- Provide real-time updates about their transport status\n`;
      systemPrompt += `- Offer emotional support and comfort\n`;
      systemPrompt += `- Answer questions about the process and timeline\n`;
      systemPrompt += `- Help with document-related questions\n`;
      systemPrompt += `- Provide guidance on next steps\n`;
      systemPrompt += `- If they ask about specific transport details, use the transport context provided\n`;
      systemPrompt += `- Be empathetic and understanding of their situation\n`;
      systemPrompt += `- If you don't have information, suggest they contact support directly\n`;
    } else if (messageType === 'emergency') {
      systemPrompt += 'This is an emergency situation. Provide immediate, clear guidance and escalate if necessary. ';
    } else if (messageType === 'documentation') {
      systemPrompt += 'Focus on documentation requirements, compliance, and proper procedures. ';
    } else if (messageType === 'support') {
      systemPrompt += 'Provide emotional support and guidance during this difficult time. ';
    }

    systemPrompt += '\n\nKeep responses concise, professional, and helpful. For family members, always be compassionate and understanding.';

    console.log('AI Chat - System Prompt:', systemPrompt);
    console.log('AI Chat - User Message:', message);

    // Generate AI response
    const completion = await openai.chat.completions.create({
      model: 'gpt-4o-mini', // Use available model instead of gpt-4
      messages: [
        {
          role: 'system',
          content: systemPrompt
        },
        {
          role: 'user',
          content: message
        }
      ],
      max_tokens: 800, // Increased for more detailed responses
      temperature: 0.7
    });

    const aiResponse = completion.choices[0].message.content;
    console.log('AI Chat - Response generated successfully');

    res.json({
      success: true,
      response: aiResponse,
      usage: completion.usage
    });
  } catch (error) {
    console.error('AI Chat Error - Full error details:', error);
    console.error('AI Chat Error - Error message:', error.message);
    console.error('AI Chat Error - Error stack:', error.stack);
    
    // Check for specific OpenAI errors
    if (error.code === 'insufficient_quota') {
      return res.status(500).json({
        success: false,
        message: 'AI service quota exceeded'
      });
    } else if (error.code === 'invalid_api_key') {
      return res.status(500).json({
        success: false,
        message: 'AI service configuration error'
      });
    } else if (error.code === 'rate_limit_exceeded') {
      return res.status(429).json({
        success: false,
        message: 'AI service rate limit exceeded'
      });
    }
    
    res.status(500).json({
      success: false,
      message: 'Failed to generate AI response',
      error: error.message
    });
  }
});

// Voice-to-voice conversation
router.post('/voice-chat', verifyToken, checkFeatureAccess('aiVoiceAgent'), upload.single('audio'), async (req, res) => {
  try {
    console.log('Voice Chat - Request received');
    console.log('Voice Chat - User:', req.user.email, 'Role:', req.user.role);
    
    const file = req.file;
    const { context, transportId } = req.body;

    console.log('Voice Chat - File received:', file ? 'Yes' : 'No');
    console.log('Voice Chat - Context:', context);
    console.log('Voice Chat - Transport ID:', transportId);

    if (!file) {
      console.log('Voice Chat - No file uploaded');
      return res.status(400).json({
        success: false,
        message: 'No audio file uploaded'
      });
    }

    console.log('Voice Chat - File path:', file.path);
    console.log('Voice Chat - File size:', file.size);

    // Step 1: Transcribe user's audio
    console.log('Voice Chat - Starting transcription...');
    let transcription = '';
    
    try {
      const transcriptionResult = await openai.audio.transcriptions.create({
        file: fs.createReadStream(file.path),
        model: 'whisper-1',
        response_format: 'text'
      });
      
      transcription = transcriptionResult;
      console.log('Voice Chat - Transcription completed:', transcription);
      
    } catch (transcriptionError) {
      console.log('Voice Chat - Transcription failed:', transcriptionError.message);
      
      // If transcription fails, use a fallback message
      if (transcriptionError.status === 429) {
        transcription = "I'm sorry, but I'm having trouble processing your voice message right now due to high demand. Please try again in a moment or use text chat instead.";
      } else {
        transcription = "I received your voice message but couldn't transcribe it. Please try again or use text chat.";
      }
      
      console.log('Voice Chat - Using fallback transcription:', transcription);
    }

    // Step 2: Generate AI response
    console.log('Voice Chat - Generating AI response...');
    
    let systemPrompt = `You are an AI voice assistant for Obitix. ${req.user.role === 'technician' ? 'Help with transport tasks professionally.' : 'Provide compassionate support to families.'} Keep responses conversational and under 30 seconds when spoken.`;
    
    // For family members, include technician voice logs context
    if (req.user.role === 'family' && transportId) {
      try {
        // Get voice logs for this transport (AI-specific endpoint)
        const voiceLogsResponse = await fetch(`${req.protocol}://${req.get('host')}/api/voice-logs/ai/family/${transportId}`, {
          headers: {
            'Authorization': req.headers.authorization,
            'Content-Type': 'application/json'
          }
        });
        
        if (voiceLogsResponse.ok) {
          const voiceLogsData = await voiceLogsResponse.json();
          if (voiceLogsData.success && voiceLogsData.data.voiceLogs.length > 0) {
                         const latestLogs = voiceLogsData.data.voiceLogs
               .filter(log => log.type === 'voice' || log.type === 'manual')
               .sort((a, b) => new Date(b.timestamp) - new Date(a.timestamp))
               .slice(0, 5); // Show more recent logs
            
             if (latestLogs.length > 0) {
               const logSummary = latestLogs.map(log => {
                 const time = new Date(log.timestamp).toLocaleTimeString();
                 const location = log.location?.address ? ` (${log.location.address})` : '';
                 const duration = log.duration ? ` (${log.duration}s)` : '';
                 return `${time}${location}${duration}: ${log.message}`;
               }).join('\n');
              
              systemPrompt += `\n\nTECHNICIAN VOICE LOGS CONTEXT (Latest ${latestLogs.length} entries):\n${logSummary}\n\nUse this information to provide accurate, real-time updates to the family member about the transport progress and any important details mentioned by the technician.`;
            }
          }
        }
      } catch (error) {
        console.log('Could not fetch voice logs for context:', error.message);
      }
    }

    let aiResponse = '';
    
    try {
      const completion = await openai.chat.completions.create({
        model: 'gpt-4o-mini',
        messages: [
          {
            role: 'system',
            content: systemPrompt
          },
          {
            role: 'user',
            content: transcription
          }
        ],
        max_tokens: 200,
        temperature: 0.7
      });

      aiResponse = completion.choices[0].message.content;
      console.log('Voice Chat - AI response generated:', aiResponse);
      
    } catch (aiError) {
      console.log('Voice Chat - AI response generation failed:', aiError.message);
      
      // If AI generation fails, use a fallback response
      if (aiError.status === 429) {
        aiResponse = "I'm experiencing high demand right now and can't process your request. Please try again in a moment or contact support directly.";
      } else {
        aiResponse = "I'm here to help you with your transport questions. Please try again or use text chat for immediate assistance.";
      }
      
      console.log('Voice Chat - Using fallback AI response:', aiResponse);
    }

    // Step 3: Convert AI response to speech (with fallback)
    console.log('Voice Chat - Converting to speech...');
    let base64Audio = null;
    
    try {
      console.log('Voice Chat - Calling OpenAI TTS with text:', aiResponse.substring(0, 100) + '...');
      
      const speech = await openai.audio.speech.create({
        model: 'tts-1',
        voice: 'alloy',
        input: aiResponse,
        response_format: 'mp3'
      });

      console.log('Voice Chat - Speech generated successfully');

      // Convert speech buffer to base64
      const buffer = Buffer.from(await speech.arrayBuffer());
      base64Audio = buffer.toString('base64');
      
      console.log('Voice Chat - Audio converted to base64, length:', base64Audio.length);
      
    } catch (ttsError) {
      console.log('Voice Chat - TTS failed:', ttsError.message);
      console.log('Voice Chat - TTS error details:', {
        status: ttsError.status,
        code: ttsError.code,
        type: ttsError.type
      });
      
      // Try alternative TTS approach
      try {
        console.log('Voice Chat - Trying alternative TTS approach...');
        const speech = await openai.audio.speech.create({
          model: 'tts-1',
          voice: 'nova',
          input: aiResponse.substring(0, 200) // Limit text length
        });
        
        const buffer = Buffer.from(await speech.arrayBuffer());
        base64Audio = buffer.toString('base64');
        console.log('Voice Chat - Alternative TTS successful, length:', base64Audio.length);
        
      } catch (altTtsError) {
        console.log('Voice Chat - Alternative TTS also failed:', altTtsError.message);
        base64Audio = 'TTS_UNAVAILABLE_' + Date.now();
        console.log('Voice Chat - Using fallback audio placeholder');
      }
    }

    // Clean up local file
    try {
      fs.unlinkSync(file.path);
      console.log('Voice Chat - Local file cleaned up');
    } catch (cleanupError) {
      console.log('Voice Chat - Error cleaning up file:', cleanupError.message);
    }

    const response = {
      success: true,
      conversation: {
        userMessage: transcription,
        aiResponse: aiResponse,
        audioResponse: base64Audio
      }
    };

    console.log('Voice Chat - Sending response with audio data');
    res.json(response);
    
  } catch (error) {
    console.error('Voice Chat - Error details:', error);
    console.error('Voice Chat - Error message:', error.message);
    console.error('Voice Chat - Error stack:', error.stack);
    
    // Clean up file if it exists
    if (req.file && req.file.path) {
      try {
        fs.unlinkSync(req.file.path);
      } catch (cleanupError) {
        console.error('Voice Chat - Error cleaning up file:', cleanupError);
      }
    }
    
    res.status(500).json({
      success: false,
      message: 'Failed to process voice chat',
      error: error.message
    });
  }
});

// Generate voice log summary
router.post('/summarize-voice-logs', verifyToken, async (req, res) => {
  try {
    const { voiceLogs, transportId } = req.body;

    if (!voiceLogs || !Array.isArray(voiceLogs)) {
      return res.status(400).json({
        success: false,
        message: 'Voice logs array is required'
      });
    }

    // Combine all voice log messages
    const combinedText = voiceLogs.map(log => log.message).join('\n');

    const systemPrompt = `You are analyzing voice logs from a funeral transport. Create a concise summary highlighting key events, issues, and important details. Focus on safety, compliance, and any notable incidents.`;

    const completion = await openai.chat.completions.create({
      model: 'gpt-4o-mini', // Use available model instead of gpt-4
      messages: [
        {
          role: 'system',
          content: systemPrompt
        },
        {
          role: 'user',
          content: `Please summarize these voice logs for transport ${transportId}:\n\n${combinedText}`
        }
      ],
      max_tokens: 300,
      temperature: 0.3
    });

    const summary = completion.choices[0].message.content;

    res.json({
      success: true,
      summary,
      logCount: voiceLogs.length
    });
  } catch (error) {
    console.error('Error summarizing voice logs:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to summarize voice logs'
    });
  }
});

// Analyze transport data with AI
router.post('/analyze-transport', verifyToken, async (req, res) => {
  try {
    const { transportData, analysisType = 'general' } = req.body;

    if (!transportData) {
      return res.status(400).json({
        success: false,
        message: 'Transport data is required'
      });
    }

    let systemPrompt = 'You are an AI analyst for Obitix transport data. ';
    
    switch (analysisType) {
      case 'safety':
        systemPrompt += 'Analyze the transport data for safety concerns, compliance issues, and risk factors. ';
        break;
      case 'efficiency':
        systemPrompt += 'Analyze the transport data for efficiency improvements, time optimization, and process enhancements. ';
        break;
      case 'compliance':
        systemPrompt += 'Analyze the transport data for regulatory compliance, documentation completeness, and legal requirements. ';
        break;
      default:
        systemPrompt += 'Provide a general analysis of the transport data. ';
    }

    const completion = await openai.chat.completions.create({
      model: 'gpt-4o-mini', // Use available model instead of gpt-4
      messages: [
        {
          role: 'system',
          content: systemPrompt
        },
        {
          role: 'user',
          content: `Please analyze this transport data:\n\n${JSON.stringify(transportData, null, 2)}`
        }
      ],
      max_tokens: 400,
      temperature: 0.3
    });

    const analysis = completion.choices[0].message.content;

    res.json({
      success: true,
      analysis,
      analysisType
    });
  } catch (error) {
    console.error('Error analyzing transport data:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to analyze transport data'
    });
  }
});

// Generate emergency response guidance
router.post('/emergency-guidance', verifyToken, async (req, res) => {
  try {
    const { emergencyType, location, transportId, description } = req.body;

    if (!emergencyType) {
      return res.status(400).json({
        success: false,
        message: 'Emergency type is required'
      });
    }

    const systemPrompt = `You are an emergency response AI for Obitix. Provide immediate, clear guidance for funeral transport emergencies. Focus on safety, legal compliance, and proper procedures. Be concise and actionable.`;

    const userPrompt = `Emergency Type: ${emergencyType}
Location: ${location || 'Unknown'}
Transport ID: ${transportId || 'Unknown'}
Description: ${description || 'No additional details'}

Please provide immediate guidance for this emergency situation.`;

    const completion = await openai.chat.completions.create({
      model: 'gpt-4o-mini', // Use available model instead of gpt-4
      messages: [
        {
          role: 'system',
          content: systemPrompt
        },
        {
          role: 'user',
          content: userPrompt
        }
      ],
      max_tokens: 300,
      temperature: 0.1
    });

    const guidance = completion.choices[0].message.content;

    res.json({
      success: true,
      guidance,
      emergencyType,
      timestamp: new Date().toISOString()
    });
  } catch (error) {
    console.error('Error generating emergency guidance:', error);
    res.status(500).json({
      success: false,
      message: 'Failed to generate emergency guidance'
    });
  }
});

module.exports = router;
