const mongoose = require('mongoose');

const qrCodeSchema = new mongoose.Schema({
  // Basic QR Code Info
  qrCodeId: {
    type: String,
    required: true,
    unique: true,
    default: () => `QR_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`
  },
  
  // Transport Case Association
  transportId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Transport',
    required: true
  },
  
  caseId: {
    type: String,
    required: true
  },
  
  // QR Code Content
  qrData: {
    type: String,
    required: true
  },
  
  // QR Code Type and Status
  type: {
    type: String,
    enum: ['pickup', 'transfer', 'delivery', 'verification'],
    required: true
  },
  
  status: {
    type: String,
    enum: ['generated', 'scanned', 'verified', 'expired'],
    default: 'generated'
  },
  
  // Location and Timing
  location: {
    type: {
      type: String,
      enum: ['Point'],
      default: 'Point'
    },
    coordinates: {
      type: [Number], // [longitude, latitude]
      required: true
    },
    address: String,
    facility: String
  },
  
  // Scan Events
  scans: [{
    scannedBy: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'User',
      required: true
    },
    scannedAt: {
      type: Date,
      default: Date.now
    },
    location: {
      type: {
        type: String,
        enum: ['Point'],
        default: 'Point'
      },
      coordinates: [Number],
      address: String
    },
    photos: [{
      photoUrl: String,
      photoId: String,
      timestamp: Date,
      metadata: {
        gps: {
          latitude: Number,
          longitude: Number,
          accuracy: Number
        },
        device: String,
        orientation: String
      }
    }],
    voiceLogs: [{
      audioUrl: String,
      transcription: String,
      duration: Number,
      timestamp: Date
    }],
    notes: String,
    verificationStatus: {
      type: String,
      enum: ['pending', 'verified', 'failed'],
      default: 'pending'
    }
  }],
  
  // Custom Branding (Enterprise Feature)
  customBranding: {
    logoUrl: String,
    customColors: {
      primary: String,
      secondary: String
    },
    funeralHomeName: String
  },
  
  // Expiration and Security
  expiresAt: {
    type: Date,
    required: true
  },
  
  isActive: {
    type: Boolean,
    default: true
  },
  
  // Organization
  organizationId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Organization',
    required: true
  },
  
  // Blockchain Integration
  blockchainHash: String,
  blockchainTransactionId: String,
  
  // Metadata
  metadata: {
    generatedBy: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'User'
    },
    generatedAt: {
      type: Date,
      default: Date.now
    },
    lastModified: {
      type: Date,
      default: Date.now
    }
  }
}, {
  timestamps: true
});

// Indexes for performance
qrCodeSchema.index({ transportId: 1, type: 1 });
qrCodeSchema.index({ qrCodeId: 1 });
qrCodeSchema.index({ organizationId: 1, status: 1 });
qrCodeSchema.index({ 'location.coordinates': '2dsphere' });
qrCodeSchema.index({ expiresAt: 1 });

// Pre-save middleware to update lastModified
qrCodeSchema.pre('save', function(next) {
  this.metadata.lastModified = new Date();
  next();
});

  // Method to generate QR code data with full transport information
  qrCodeSchema.methods.generateQRData = async function() {
    try {
      // Get transport details
      const Transport = require('./Transport');
      const transport = await Transport.findById(this.transportId).populate('assignedTechnician', 'firstName lastName');
      
      if (!transport) {
        throw new Error('Transport not found');
      }

      const data = {
        qrCodeId: this.qrCodeId,
        transportId: this.transportId.toString(),
        caseId: this.caseId,
        deceasedName: transport.deceased?.firstName && transport.deceased?.lastName 
          ? `${transport.deceased.firstName} ${transport.deceased.lastName}` 
          : transport.deceasedName || 'Unknown',
        pickupLocation: {
          address: transport.pickupLocation?.address || '',
          coordinates: transport.pickupLocation?.coordinates || [0, 0],
          facility: transport.pickupLocation?.facility || ''
        },
        deliveryLocation: {
          address: transport.deliveryLocation?.address || '',
          coordinates: transport.deliveryLocation?.coordinates || [0, 0],
          facility: transport.deliveryLocation?.facility || ''
        },
        technicianName: transport.technicianName || 
          (transport.assignedTechnician ? `${transport.assignedTechnician.firstName || ''} ${transport.assignedTechnician.lastName || ''}`.trim() : 'Unassigned'),
        technicianId: transport.assignedTechnician?._id?.toString() || '',
        status: transport.status || 'scheduled',
        scheduledPickupTime: transport.scheduledPickupTime,
        estimatedDeliveryTime: transport.estimatedDeliveryTime,
        organizationId: this.organizationId.toString(),
        organizationName: transport.organizationName || 'Obitix Funeral Services',
        timestamp: this.metadata.generatedAt,
        type: this.type
      };
      
      return JSON.stringify(data);
    } catch (error) {
      console.error('Error generating QR code data:', error);
      // Fallback to basic data if transport not found
      const basicData = {
        qrCodeId: this.qrCodeId,
        transportId: this.transportId.toString(),
        caseId: this.caseId,
        type: this.type,
        timestamp: this.metadata.generatedAt,
        organizationId: this.organizationId.toString()
      };
      return JSON.stringify(basicData);
    }
  };

// Method to verify QR code
qrCodeSchema.methods.verifyQRCode = function() {
  const now = new Date();
  
  // Check if expired
  if (now > this.expiresAt) {
    this.status = 'expired';
    return { valid: false, reason: 'QR code expired' };
  }
  
  // Check if active
  if (!this.isActive) {
    return { valid: false, reason: 'QR code inactive' };
  }
  
  return { valid: true };
};

// Method to add scan event
qrCodeSchema.methods.addScan = function(scanData) {
  this.scans.push(scanData);
  this.status = 'scanned';
  this.metadata.lastModified = new Date();
  return this.save();
};

// Static method to find QR codes by transport
qrCodeSchema.statics.findByTransport = function(transportId) {
  return this.find({ transportId, isActive: true }).sort({ 'metadata.generatedAt': 1 });
};

// Static method to find active QR codes
qrCodeSchema.statics.findActive = function(organizationId) {
  return this.find({ 
    organizationId, 
    isActive: true, 
    expiresAt: { $gt: new Date() } 
  });
};

module.exports = mongoose.model('QRCode', qrCodeSchema);
