const mongoose = require('mongoose');

const documentSchema = new mongoose.Schema({
  // Basic Information
  filename: {
    type: String,
    required: true,
    trim: true
  },
  originalName: {
    type: String,
    required: true,
    trim: true
  },
  mimeType: {
    type: String,
    required: true
  },
  size: {
    type: Number,
    required: true
  },
  
  // Storage Information
  storageType: {
    type: String,
    enum: ['local', 's3', 'ipfs'],
    default: 'local'
  },
  storagePath: {
    type: String,
    required: true
  },
  ipfsHash: String,
  s3Key: String,
  
  // Document Metadata
  documentType: {
    type: String,
    enum: [
      // Family Member Documents
      'death_certificate',
      'medical_examiner_report',
      'family_id_document',
      'family_contact_info',
      'special_instructions',
      'legal_documents',
      'power_of_attorney',
      'executor_documents',
      
      // Funeral Home Staff Documents
      'transport_permit',
      'chain_of_custody_form',
      'compliance_certificate',
      'insurance_document',
      'liability_document',
      'route_planning',
      'emergency_contacts',
      'regulatory_compliance',
      
      // Technician Documents
      'checkpoint_photo',
      'condition_report',
      'incident_report',
      'completion_certificate',
      'gps_tracking_log',
      'pickup_verification',
      'delivery_verification',
      
      // Medical Facility Documents
      'release_form',
      'medical_records',
      'transfer_authorization',
      'facility_documentation',
      
      // General Documents
      'family_consent',
      'cremation_authorization',
      'burial_permit',
      'family_photo',
      'custody_verification',
      'other'
    ],
    required: true
  },
  description: {
    type: String,
    trim: true
  },
  tags: [String],
  
  // Access Control
  organizationId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Organization',
    required: true,
    index: true
  },
  transportId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Transport',
    index: true
  },
  uploadedBy: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
    required: true
  },
  
  // Role-specific fields
  familyId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
    index: true
  },
  technicianId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User',
    index: true
  },
  medicalFacilityId: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'Organization',
    index: true
  },
  
  // Permissions
  isPublic: {
    type: Boolean,
    default: false
  },
  allowedRoles: [{
    type: String,
    enum: ['family', 'technician', 'staff', 'admin']
  }],
  allowedUsers: [{
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User'
  }],
  
  // Security
  isEncrypted: {
    type: Boolean,
    default: true
  },
  encryptionKey: String,
  checksum: {
    type: String,
    required: true
  },
  
  // Status
  status: {
    type: String,
    enum: ['uploading', 'processing', 'pending_approval', 'approved', 'rejected', 'expired', 'ready', 'error', 'deleted'],
    default: 'uploading'
  },
  processingError: String,
  
  // Approval and Validation
  approvalStatus: {
    type: String,
    enum: ['pending', 'approved', 'rejected', 'requires_revision'],
    default: 'pending'
  },
  approvedBy: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User'
  },
  approvedAt: Date,
  rejectionReason: String,
  requiresRevision: {
    type: Boolean,
    default: false
  },
  revisionNotes: String,
  
  // Document Validation
  isVerified: {
    type: Boolean,
    default: false
  },
  verifiedBy: {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User'
  },
  verifiedAt: Date,
  verificationNotes: String,
  
  // Audit Trail
  createdAt: {
    type: Date,
    default: Date.now
  },
  updatedAt: {
    type: Date,
    default: Date.now
  },
  deletedAt: Date,
  
  // Version Control
  version: {
    type: Number,
    default: 1
  },
  previousVersions: [{
    version: Number,
    storagePath: String,
    ipfsHash: String,
    updatedAt: Date,
    updatedBy: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'User'
    }
  }],
  
  // Compliance
  retentionPolicy: {
    type: String,
    enum: ['7_years', 'permanent', 'custom'],
    default: '7_years'
  },
  retentionExpiry: Date,
  isCompliant: {
    type: Boolean,
    default: true
  },
  
  // Family Access
  familyAccess: {
    enabled: {
      type: Boolean,
      default: false
    },
    sharedAt: Date,
    sharedBy: {
      type: mongoose.Schema.Types.ObjectId,
      ref: 'User'
    },
    accessExpiry: Date
  }
}, {
  timestamps: true
});

// Indexes for performance
documentSchema.index({ organizationId: 1, createdAt: -1 });
documentSchema.index({ transportId: 1, documentType: 1 });
documentSchema.index({ uploadedBy: 1, createdAt: -1 });
documentSchema.index({ status: 1, organizationId: 1 });
documentSchema.index({ 'familyAccess.enabled': 1, organizationId: 1 });

// Virtual for full file path
documentSchema.virtual('fullPath').get(function() {
  if (this.storageType === 'ipfs' && this.ipfsHash) {
    return `ipfs://${this.ipfsHash}`;
  }
  if (this.storageType === 's3' && this.s3Key) {
    return `s3://${this.s3Key}`;
  }
  return this.storagePath;
});

// Virtual for file extension
documentSchema.virtual('extension').get(function() {
  return this.originalName.split('.').pop().toLowerCase();
});

// Virtual for human readable size
documentSchema.virtual('humanSize').get(function() {
  const bytes = this.size;
  const sizes = ['Bytes', 'KB', 'MB', 'GB'];
  if (bytes === 0) return '0 Bytes';
  const i = Math.floor(Math.log(bytes) / Math.log(1024));
  return Math.round(bytes / Math.pow(1024, i) * 100) / 100 + ' ' + sizes[i];
});

// Pre-save middleware
documentSchema.pre('save', function(next) {
  this.updatedAt = new Date();
  
  // Generate checksum if not provided
  if (!this.checksum && this.storagePath) {
    // In a real implementation, you would calculate the actual file checksum
    this.checksum = require('crypto').createHash('sha256').update(this.storagePath).digest('hex');
  }
  
  next();
});

// Static method to find documents by organization
documentSchema.statics.findByOrganization = function(organizationId, options = {}) {
  const query = { organizationId };
  
  if (options.status) {
    query.status = options.status;
  }
  
  if (options.documentType) {
    query.documentType = options.documentType;
  }
  
  if (options.transportId) {
    query.transportId = options.transportId;
  }
  
  return this.find(query)
    .populate('uploadedBy', 'firstName lastName email')
    .populate('transportId', 'caseId deceased.firstName deceased.lastName')
    .sort({ createdAt: -1 });
};

// Static method to find documents accessible by user
documentSchema.statics.findAccessibleByUser = function(userId, userRole, organizationId, options = {}) {
  const query = {
    organizationId,
    $or: [
      { isPublic: true },
      { allowedRoles: userRole },
      { allowedUsers: userId },
      { uploadedBy: userId }
    ]
  };
  
  if (options.transportId) {
    query.transportId = options.transportId;
  }
  
  if (options.documentType) {
    query.documentType = options.documentType;
  }
  
  return this.find(query)
    .populate('uploadedBy', 'firstName lastName email')
    .populate('transportId', 'caseId deceased.firstName deceased.lastName')
    .sort({ createdAt: -1 });
};

// Instance method to check if user can access document
documentSchema.methods.canAccess = function(userId, userRole) {
  // Document is public
  if (this.isPublic) return true;
  
  // User uploaded the document
  if (this.uploadedBy.toString() === userId.toString()) return true;
  
  // User role is allowed
  if (this.allowedRoles.includes(userRole)) return true;
  
  // User is specifically allowed
  if (this.allowedUsers.some(id => id.toString() === userId.toString())) return true;
  
  return false;
};

// Instance method to share with family
documentSchema.methods.shareWithFamily = function(userId, expiryDate = null) {
  this.familyAccess = {
    enabled: true,
    sharedAt: new Date(),
    sharedBy: userId,
    accessExpiry: expiryDate
  };
  
  return this.save();
};

// Instance method to revoke family access
documentSchema.methods.revokeFamilyAccess = function() {
  this.familyAccess = {
    enabled: false,
    sharedAt: null,
    sharedBy: null,
    accessExpiry: null
  };
  
  return this.save();
};

// Instance method to create new version
documentSchema.methods.createVersion = function(newStoragePath, newIpfsHash, userId) {
  this.previousVersions.push({
    version: this.version,
    storagePath: this.storagePath,
    ipfsHash: this.ipfsHash,
    updatedAt: this.updatedAt,
    updatedBy: this.uploadedBy
  });
  
  this.version += 1;
  this.storagePath = newStoragePath;
  this.ipfsHash = newIpfsHash;
  this.uploadedBy = userId;
  
  return this.save();
};

// Instance method to soft delete
documentSchema.methods.softDelete = function() {
  this.status = 'deleted';
  this.deletedAt = new Date();
  return this.save();
};

// Instance method to restore
documentSchema.methods.restore = function() {
  this.status = 'ready';
  this.deletedAt = undefined;
  return this.save();
};

module.exports = mongoose.model('Document', documentSchema);
