const jwt = require('jsonwebtoken');
const User = require('../models/User');
const Organization = require('../models/Organization');

// Verify JWT token
const verifyToken = async (req, res, next) => {
  try {
    let token;
    
    // Check for token in headers
    if (req.headers.authorization && req.headers.authorization.startsWith('Bearer')) {
      token = req.headers.authorization.split(' ')[1];
    }
    // Check for token in cookies
    else if (req.cookies && req.cookies.token) {
      token = req.cookies.token;
    }
    
    if (!token) {
      return res.status(401).json({
        success: false,
        message: 'Access denied. No token provided.'
      });
    }
    
    // Verify token
    const decoded = jwt.verify(token, process.env.JWT_SECRET);
    
    // Check if user still exists
    const user = await User.findById(decoded.id).select('-password');
    console.log('Token verification - Decoded user ID:', decoded.id);
    console.log('Token verification - Found user:', user ? user.role : 'Not found');
    
    if (!user) {
      return res.status(401).json({
        success: false,
        message: 'User no longer exists.'
      });
    }
    
    // Check if user is active
    if (!user.isActive) {
      return res.status(401).json({
        success: false,
        message: 'User account is deactivated.'
      });
    }
    
    // Check if user is locked
    if (user.isLocked) {
      return res.status(401).json({
        success: false,
        message: 'User account is temporarily locked.'
      });
    }
    
    // Check if password was changed after token was issued
    if (user.changedPasswordAfter(decoded.iat)) {
      return res.status(401).json({
        success: false,
        message: 'User recently changed password. Please log in again.'
      });
    }
    
    // Add user to request object
    req.user = user;
    next();
  } catch (error) {
    if (error.name === 'JsonWebTokenError') {
      return res.status(401).json({
        success: false,
        message: 'Invalid token.'
      });
    }
    if (error.name === 'TokenExpiredError') {
      return res.status(401).json({
        success: false,
        message: 'Token expired.'
      });
    }
    
    return res.status(500).json({
      success: false,
      message: 'Authentication error.'
    });
  }
};

// Role-based access control
const authorize = (...roles) => {
  return (req, res, next) => {
    if (!req.user) {
      return res.status(401).json({
        success: false,
        message: 'Authentication required.'
      });
    }
    
    // Flatten the roles array to handle nested arrays
    const flatRoles = roles.flat();
    
    
    if (!flatRoles.includes(req.user.role)) {
      console.log('Authorization failed - User role not in required roles');
      console.log('Authorization failed - User role value:', JSON.stringify(req.user.role));
      console.log('Authorization failed - Required roles:', JSON.stringify(flatRoles));
      return res.status(403).json({
        success: false,
        message: `User role '${req.user.role}' is not authorized to access this resource.`
      });
    }
    
    console.log('Authorization successful');
    next();
  };
};

// Permission-based access control
const requirePermission = (permission) => {
  return (req, res, next) => {
    if (!req.user) {
      return res.status(401).json({
        success: false,
        message: 'Authentication required.'
      });
    }
    
    if (!req.user.permissions.includes(permission)) {
      return res.status(403).json({
        success: false,
        message: `Permission '${permission}' is required to access this resource.`
      });
    }
    
    next();
  };
};

// Organization access control
const requireOrganizationAccess = async (req, res, next) => {
  try {
    if (!req.user) {
      return res.status(401).json({
        success: false,
        message: 'Authentication required.'
      });
    }
    
    // Family members don't need organization access
    if (req.user.role === 'family') {
      return next();
    }
    
    // Admin can access all organizations
    if (req.user.role === 'admin') {
      return next();
    }
    
    // Check if user has organization access
    if (!req.user.organizationId) {
      return res.status(403).json({
        success: false,
        message: 'Organization access required.'
      });
    }
    
    // Check if organization is active
    const organization = await Organization.findById(req.user.organizationId);
    if (!organization || organization.status !== 'active') {
      return res.status(403).json({
        success: false,
        message: 'Organization is not active.'
      });
    }
    
    // Check if organization subscription is active
    if (!organization.isSubscriptionActive) {
      return res.status(403).json({
        success: false,
        message: 'Organization subscription is not active.'
      });
    }
    
    req.organization = organization;
    next();
  } catch (error) {
    return res.status(500).json({
      success: false,
      message: 'Organization access verification failed.'
    });
  }
};

// Resource ownership verification
const verifyResourceOwnership = (resourceModel, resourceIdField = 'id') => {
  return async (req, res, next) => {
    try {
      const resourceId = req.params[resourceIdField];
      const resource = await resourceModel.findById(resourceId);
      
      if (!resource) {
        return res.status(404).json({
          success: false,
          message: 'Resource not found.'
        });
      }
      
      // Admin can access all resources
      if (req.user.role === 'admin') {
        req.resource = resource;
        return next();
      }
      
      // Check if user owns the resource or has organization access
      if (resource.userId && resource.userId.toString() === req.user._id.toString()) {
        req.resource = resource;
        return next();
      }
      
      if (resource.organizationId && resource.organizationId.toString() === req.user.organizationId?.toString()) {
        req.resource = resource;
        return next();
      }
      
      return res.status(403).json({
        success: false,
        message: 'Access denied. You do not have permission to access this resource.'
      });
    } catch (error) {
      return res.status(500).json({
        success: false,
        message: 'Resource ownership verification failed.'
      });
    }
  };
};

// Rate limiting for authentication attempts
const authRateLimit = (req, res, next) => {
  // Rate limiting disabled for development
  next();
};

// Two-factor authentication middleware
const require2FA = async (req, res, next) => {
  try {
    if (!req.user) {
      return res.status(401).json({
        success: false,
        message: 'Authentication required.'
      });
    }
    
    if (req.user.twoFactorEnabled) {
      const twoFactorToken = req.headers['x-2fa-token'] || req.body.twoFactorToken;
      
      if (!twoFactorToken) {
        return res.status(401).json({
          success: false,
          message: 'Two-factor authentication token required.',
          requires2FA: true
        });
      }
      
      // Verify 2FA token (implementation would depend on the 2FA method used)
      // For now, we'll just check if it's provided
      if (twoFactorToken.length < 6) {
        return res.status(401).json({
          success: false,
          message: 'Invalid two-factor authentication token.'
        });
      }
    }
    
    next();
  } catch (error) {
    return res.status(500).json({
      success: false,
      message: 'Two-factor authentication verification failed.'
    });
  }
};

// Audit logging middleware
const auditLog = (action) => {
  return (req, res, next) => {
    const originalSend = res.send;
    
    res.send = function(data) {
      // Log the action after response is sent
      if (req.user) {
        // This would typically save to an audit log collection
        console.log(`AUDIT: User ${req.user._id} performed ${action} at ${new Date().toISOString()}`);
      }
      
      originalSend.call(this, data);
    };
    
    next();
  };
};

module.exports = {
  verifyToken,
  authorize,
  requirePermission,
  requireOrganizationAccess,
  verifyResourceOwnership,
  authRateLimit,
  require2FA,
  auditLog
};
