const express = require('express');
const cors = require('cors');
const helmet = require('helmet');
const rateLimit = require('express-rate-limit');
const compression = require('compression');
const session = require('express-session');
const MongoStore = require('connect-mongo');
const passport = require('passport');
const mongoose = require('mongoose');
const winston = require('winston');
const dotenv = require('dotenv');
const path = require('path');

// Load environment variables
dotenv.config();
// Database connection
mongoose.connect(process.env.MONGODB_URI)
.then(() => {
  logger.info('Connected to MongoDB');
})
.catch((error) => {
  logger.error('MongoDB connection error:', error);
  process.exit(1);
});

// Import routes
const authRoutes = require('./routes/auth');
const userRoutes = require('./routes/users');
const transportRoutes = require('./routes/transports');
const blockchainRoutes = require('./routes/blockchain');
const aiRoutes = require('./routes/ai');
const documentRoutes = require('./routes/documents');
const notificationRoutes = require('./routes/notifications');
const organizationRoutes = require('./routes/organizations');
const adminRoutes = require('./routes/admin');
const subscriptionRoutes = require('./routes/subscriptions');
const voiceLogRoutes = require('./routes/voice-logs');

// Import middleware
const { errorHandler } = require('./middleware/errorHandler');
const { verifyToken } = require('./middleware/auth');
const { rateLimiter } = require('./middleware/rateLimiter');

// Import services
const { initializeBlockchain } = require('./services/blockchain');
const { initializeAI } = require('./services/ai');
const { initializeSocket } = require('./services/socket');
const EmailService = require('./services/email');

// Initialize logger
const logger = winston.createLogger({
  level: 'info',
  format: winston.format.combine(
    winston.format.timestamp(),
    winston.format.errors({ stack: true }),
    winston.format.json()
  ),
  defaultMeta: { service: 'obitix-server' },
  transports: [
    new winston.transports.File({ filename: 'logs/error.log', level: 'error' }),
    new winston.transports.File({ filename: 'logs/combined.log' }),
  ],
});

if (process.env.NODE_ENV !== 'production') {
  logger.add(new winston.transports.Console({
    format: winston.format.simple()
  }));
}

const app = express();
const PORT = process.env.PORT || 5000;

// Security middleware
app.use(helmet({
  contentSecurityPolicy: {
    directives: {
      defaultSrc: ["'self'"],
      styleSrc: ["'self'", "'unsafe-inline'"],
      scriptSrc: ["'self'"],
      imgSrc: ["'self'", "data:", "https:"],
    },
  },
}));

// Rate limiting
const limiter = rateLimit({
  windowMs: 15 * 60 * 1000, // 15 minutes
  max: 100, // limit each IP to 100 requests per windowMs
  message: 'Too many requests from this IP, please try again later.',
  standardHeaders: true,
  legacyHeaders: false,
});
app.use('/api/', limiter);

// Body parsing middleware
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true, limit: '10mb' }));

// Compression middleware
app.use(compression());

// CORS configuration
app.use(cors({
  origin: '*', // Allow all origins
  credentials: false, // Disable credentials when using wildcard origin
  methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
  allowedHeaders: ['Content-Type', 'Authorization', 'X-Requested-With'],
}));

// Session configuration
app.use(session({
  secret: process.env.SESSION_SECRET || 'obitix-secret-key',
  resave: false,
  saveUninitialized: false,
  store: MongoStore.create({
    mongoUrl: process.env.MONGODB_URI,
    ttl: 24 * 60 * 60, // 1 day
  }),
  cookie: {
    secure: process.env.NODE_ENV === 'production',
    httpOnly: true,
    maxAge: 24 * 60 * 60 * 1000, // 1 day
  },
}));

// Passport middleware
app.use(passport.initialize());
app.use(passport.session());

// Static files
app.use('/uploads', express.static(path.join(__dirname, '../uploads')));

// Health check endpoint
app.get('/health', (req, res) => {
  res.status(200).json({
    status: 'OK',
    timestamp: new Date().toISOString(),
    uptime: process.uptime(),
    environment: process.env.NODE_ENV,
  });
});

// API routes
app.use('/api/auth', authRoutes);
app.use('/api/users', verifyToken, userRoutes);
app.use('/api/transports', verifyToken, transportRoutes);
app.use('/api/blockchain', verifyToken, blockchainRoutes);
app.use('/api/ai', verifyToken, aiRoutes);
app.use('/api/documents', verifyToken, documentRoutes);
app.use('/api/notifications', verifyToken, notificationRoutes);
app.use('/api/organizations', verifyToken, organizationRoutes);
app.use('/api/admin', verifyToken, adminRoutes);
app.use('/api/subscriptions', subscriptionRoutes);
app.use('/api/voice-logs', verifyToken, voiceLogRoutes);
app.use('/api/qr-codes', verifyToken, require('./routes/qr-codes'));
app.use('/api/transport-status', require('./routes/transport-status'));

// Error handling middleware
app.use(errorHandler);

// 404 handler
app.use('*', (req, res) => {
  res.status(404).json({
    success: false,
    message: 'Route not found',
    path: req.originalUrl,
  });
});



// MongoDB-only configuration
logger.info('Using MongoDB for session storage and data persistence');

// Initialize services
async function initializeServices() {
  try {
    // Initialize blockchain (optional)
    try {
      await initializeBlockchain();
    } catch (error) {
      logger.warn('Blockchain initialization failed, continuing without blockchain support');
    }
    
    // Initialize AI service (optional)
    try {
      await initializeAI();
    } catch (error) {
      logger.warn('AI initialization failed, continuing without AI support');
    }
    
    // Initialize email service (optional)
    try {
      if (process.env.SMTP_HOST && process.env.SMTP_USER && process.env.SMTP_PASS) {
        await EmailService.initialize();
        logger.info('Email service initialized successfully');
      } else {
        logger.warn('Email configuration not found, continuing without email support');
      }
    } catch (error) {
      logger.warn('Email service initialization failed, continuing without email support');
    }
    
    // Start server - listen on all interfaces for mobile development
    const server = app.listen(PORT, '0.0.0.0', () => {
      logger.info(`Obitix server running on port ${PORT} (accessible from all interfaces)`);
      logger.info(`Environment: ${process.env.NODE_ENV}`);
    });

    // Initialize Socket.IO
    initializeSocket(server);

  } catch (error) {
    logger.error('Failed to initialize services:', error);
    process.exit(1);
  }
}

// Graceful shutdown
process.on('SIGTERM', () => {
  logger.info('SIGTERM received, shutting down gracefully');
  mongoose.connection.close();
  process.exit(0);
});

process.on('SIGINT', () => {
  logger.info('SIGINT received, shutting down gracefully');
  mongoose.connection.close();
  process.exit(0);
});

// Start the application
initializeServices();

module.exports = app;
