// SPDX-License-Identifier: MIT
pragma solidity ^0.8.19;

import "@openzeppelin/contracts/access/Ownable.sol";
import "@openzeppelin/contracts/security/Pausable.sol";
import "@openzeppelin/contracts/security/ReentrancyGuard.sol";
import "@openzeppelin/contracts/utils/Counters.sol";

/**
 * @title ObitixCustody
 * @dev Smart contract for tracking chain-of-custody events in funeral transport
 * @author Obitix Team
 */
contract ObitixCustody is Ownable, Pausable, ReentrancyGuard {
    using Counters for Counters.Counter;

    // Structs
    struct CustodyEvent {
        string eventType;
        string metadata;
        uint256 timestamp;
        address recordedBy;
        bool isValid;
    }

    struct Transport {
        string caseId;
        string uniqueIdTag;
        address organization;
        uint256 createdAt;
        uint256 lastUpdated;
        bool isActive;
        uint256 eventCount;
    }

    struct Organization {
        string name;
        string licenseNumber;
        address walletAddress;
        bool isVerified;
        uint256 registeredAt;
        uint256 transportCount;
    }

    // State variables
    Counters.Counter private _eventIds;
    Counters.Counter private _transportIds;
    Counters.Counter private _organizationIds;

    // Mappings
    mapping(uint256 => CustodyEvent) public custodyEvents;
    mapping(string => Transport) public transports; // caseId => Transport
    mapping(address => Organization) public organizations;
    mapping(address => bool) public authorizedOperators;
    mapping(string => uint256[]) public transportEvents; // caseId => eventIds
    mapping(bytes32 => bool) public eventHashes;

    // Events
    event CustodyEventRecorded(
        uint256 indexed eventId,
        string indexed caseId,
        string eventType,
        string metadata,
        uint256 timestamp,
        address recordedBy
    );

    event TransportCreated(
        string indexed caseId,
        string uniqueIdTag,
        address organization,
        uint256 timestamp
    );

    event OrganizationRegistered(
        address indexed walletAddress,
        string name,
        string licenseNumber,
        uint256 timestamp
    );

    event OrganizationVerified(
        address indexed walletAddress,
        bool isVerified,
        uint256 timestamp
    );

    event OperatorAuthorized(
        address indexed operator,
        bool isAuthorized,
        uint256 timestamp
    );

    event ContractPaused(address indexed by, uint256 timestamp);
    event ContractUnpaused(address indexed by, uint256 timestamp);

    // Modifiers
    modifier onlyAuthorized() {
        require(
            msg.sender == owner() || authorizedOperators[msg.sender],
            "ObitixCustody: Not authorized"
        );
        _;
    }

    modifier onlyVerifiedOrganization() {
        require(
            organizations[msg.sender].isVerified,
            "ObitixCustody: Organization not verified"
        );
        _;
    }

    modifier transportExists(string memory caseId) {
        require(
            transports[caseId].createdAt != 0,
            "ObitixCustody: Transport does not exist"
        );
        _;
    }

    modifier transportActive(string memory caseId) {
        require(
            transports[caseId].isActive,
            "ObitixCustody: Transport is not active"
        );
        _;
    }

    // Constructor
    constructor() {
        _eventIds.increment(); // Start from 1
        _transportIds.increment();
        _organizationIds.increment();
    }

    /**
     * @dev Record a custody event for a transport
     * @param caseId The unique case identifier
     * @param eventType The type of custody event
     * @param metadata JSON string containing event details
     */
    function recordCustodyEvent(
        string memory caseId,
        string memory eventType,
        string memory metadata
    ) external onlyAuthorized whenNotPaused nonReentrant returns (bytes32) {
        require(bytes(caseId).length > 0, "ObitixCustody: Invalid case ID");
        require(bytes(eventType).length > 0, "ObitixCustody: Invalid event type");
        require(bytes(metadata).length > 0, "ObitixCustody: Invalid metadata");

        // Verify transport exists and is active
        require(
            transports[caseId].createdAt != 0,
            "ObitixCustody: Transport does not exist"
        );
        require(
            transports[caseId].isActive,
            "ObitixCustody: Transport is not active"
        );

        // Create event hash to prevent duplicates
        bytes32 eventHash = keccak256(
            abi.encodePacked(caseId, eventType, metadata, block.timestamp)
        );
        require(!eventHashes[eventHash], "ObitixCustody: Duplicate event");

        uint256 eventId = _eventIds.current();
        _eventIds.increment();

        // Create custody event
        CustodyEvent memory newEvent = CustodyEvent({
            eventType: eventType,
            metadata: metadata,
            timestamp: block.timestamp,
            recordedBy: msg.sender,
            isValid: true
        });

        custodyEvents[eventId] = newEvent;
        eventHashes[eventHash] = true;

        // Add event to transport
        transportEvents[caseId].push(eventId);
        transports[caseId].eventCount++;
        transports[caseId].lastUpdated = block.timestamp;

        emit CustodyEventRecorded(
            eventId,
            caseId,
            eventType,
            metadata,
            block.timestamp,
            msg.sender
        );

        return eventHash;
    }

    /**
     * @dev Create a new transport case
     * @param caseId The unique case identifier
     * @param uniqueIdTag The unique ID tag for the deceased
     * @param organization The organization address
     */
    function createTransport(
        string memory caseId,
        string memory uniqueIdTag,
        address organization
    ) external onlyAuthorized whenNotPaused nonReentrant {
        require(bytes(caseId).length > 0, "ObitixCustody: Invalid case ID");
        require(bytes(uniqueIdTag).length > 0, "ObitixCustody: Invalid ID tag");
        require(organization != address(0), "ObitixCustody: Invalid organization");
        require(
            transports[caseId].createdAt == 0,
            "ObitixCustody: Transport already exists"
        );

        Transport memory newTransport = Transport({
            caseId: caseId,
            uniqueIdTag: uniqueIdTag,
            organization: organization,
            createdAt: block.timestamp,
            lastUpdated: block.timestamp,
            isActive: true,
            eventCount: 0
        });

        transports[caseId] = newTransport;
        organizations[organization].transportCount++;

        emit TransportCreated(caseId, uniqueIdTag, organization, block.timestamp);
    }

    /**
     * @dev Register a new organization
     * @param name Organization name
     * @param licenseNumber Business license number
     */
    function registerOrganization(
        string memory name,
        string memory licenseNumber
    ) external whenNotPaused nonReentrant {
        require(bytes(name).length > 0, "ObitixCustody: Invalid name");
        require(
            bytes(licenseNumber).length > 0,
            "ObitixCustody: Invalid license number"
        );
        require(
            organizations[msg.sender].registeredAt == 0,
            "ObitixCustody: Organization already registered"
        );

        Organization memory newOrg = Organization({
            name: name,
            licenseNumber: licenseNumber,
            walletAddress: msg.sender,
            isVerified: false,
            registeredAt: block.timestamp,
            transportCount: 0
        });

        organizations[msg.sender] = newOrg;

        emit OrganizationRegistered(
            msg.sender,
            name,
            licenseNumber,
            block.timestamp
        );
    }

    /**
     * @dev Get all custody events for a transport
     * @param caseId The case identifier
     * @return Array of custody events
     */
    function getCustodyEvents(
        string memory caseId
    ) external view returns (CustodyEvent[] memory) {
        require(
            transports[caseId].createdAt != 0,
            "ObitixCustody: Transport does not exist"
        );

        uint256[] memory eventIds = transportEvents[caseId];
        CustodyEvent[] memory events = new CustodyEvent[](eventIds.length);

        for (uint256 i = 0; i < eventIds.length; i++) {
            events[i] = custodyEvents[eventIds[i]];
        }

        return events;
    }

    /**
     * @dev Get transport information
     * @param caseId The case identifier
     * @return Transport information
     */
    function getTransport(
        string memory caseId
    ) external view returns (Transport memory) {
        require(
            transports[caseId].createdAt != 0,
            "ObitixCustody: Transport does not exist"
        );
        return transports[caseId];
    }

    /**
     * @dev Get organization information
     * @param walletAddress The organization wallet address
     * @return Organization information
     */
    function getOrganization(
        address walletAddress
    ) external view returns (Organization memory) {
        require(
            organizations[walletAddress].registeredAt != 0,
            "ObitixCustody: Organization does not exist"
        );
        return organizations[walletAddress];
    }

    /**
     * @dev Verify an organization (owner only)
     * @param walletAddress The organization wallet address
     * @param isVerified Verification status
     */
    function verifyOrganization(
        address walletAddress,
        bool isVerified
    ) external onlyOwner {
        require(
            organizations[walletAddress].registeredAt != 0,
            "ObitixCustody: Organization does not exist"
        );

        organizations[walletAddress].isVerified = isVerified;

        emit OrganizationVerified(walletAddress, isVerified, block.timestamp);
    }

    /**
     * @dev Authorize or revoke operator access (owner only)
     * @param operator The operator address
     * @param isAuthorized Authorization status
     */
    function setOperatorAuthorization(
        address operator,
        bool isAuthorized
    ) external onlyOwner {
        authorizedOperators[operator] = isAuthorized;

        emit OperatorAuthorized(operator, isAuthorized, block.timestamp);
    }

    /**
     * @dev Deactivate a transport
     * @param caseId The case identifier
     */
    function deactivateTransport(
        string memory caseId
    ) external onlyAuthorized transportExists(caseId) transportActive(caseId) {
        transports[caseId].isActive = false;
        transports[caseId].lastUpdated = block.timestamp;
    }

    /**
     * @dev Reactivate a transport
     * @param caseId The case identifier
     */
    function reactivateTransport(
        string memory caseId
    ) external onlyAuthorized transportExists(caseId) {
        require(
            !transports[caseId].isActive,
            "ObitixCustody: Transport is already active"
        );

        transports[caseId].isActive = true;
        transports[caseId].lastUpdated = block.timestamp;
    }

    /**
     * @dev Pause the contract (owner only)
     */
    function pause() external onlyOwner {
        _pause();
        emit ContractPaused(msg.sender, block.timestamp);
    }

    /**
     * @dev Unpause the contract (owner only)
     */
    function unpause() external onlyOwner {
        _unpause();
        emit ContractUnpaused(msg.sender, block.timestamp);
    }

    /**
     * @dev Get contract statistics
     * @return Total events, transports, and organizations
     */
    function getContractStats()
        external
        view
        returns (uint256 totalEvents, uint256 totalTransports, uint256 totalOrganizations)
    {
        return (
            _eventIds.current() - 1,
            _transportIds.current() - 1,
            _organizationIds.current() - 1
        );
    }

    /**
     * @dev Check if an address is an authorized operator
     * @param operator The address to check
     * @return True if authorized
     */
    function isAuthorizedOperator(
        address operator
    ) external view returns (bool) {
        return authorizedOperators[operator] || operator == owner();
    }

    /**
     * @dev Check if a transport exists and is active
     * @param caseId The case identifier
     * @return True if transport exists and is active
     */
    function isTransportActive(
        string memory caseId
    ) external view returns (bool) {
        return transports[caseId].createdAt != 0 && transports[caseId].isActive;
    }

    /**
     * @dev Get the number of events for a transport
     * @param caseId The case identifier
     * @return Number of events
     */
    function getTransportEventCount(
        string memory caseId
    ) external view returns (uint256) {
        return transportEvents[caseId].length;
    }

    /**
     * @dev Emergency function to invalidate an event (owner only)
     * @param eventId The event ID to invalidate
     */
    function invalidateEvent(uint256 eventId) external onlyOwner {
        require(
            custodyEvents[eventId].timestamp != 0,
            "ObitixCustody: Event does not exist"
        );
        require(
            custodyEvents[eventId].isValid,
            "ObitixCustody: Event already invalidated"
        );

        custodyEvents[eventId].isValid = false;
    }

    /**
     * @dev Withdraw any ETH sent to the contract (owner only)
     */
    function withdraw() external onlyOwner {
        uint256 balance = address(this).balance;
        require(balance > 0, "ObitixCustody: No balance to withdraw");

        (bool success, ) = payable(owner()).call{value: balance}("");
        require(success, "ObitixCustody: Withdrawal failed");
    }

    // Fallback function
    receive() external payable {
        // Accept ETH payments
    }
}
