const mongoose = require('mongoose');
const winston = require('winston');

// MongoDB Atlas connection options
const mongoOptions = {
  useNewUrlParser: true,
  useUnifiedTopology: true,
  ssl: true,
  sslValidate: false, // Disable SSL certificate validation
  retryWrites: true,
  w: 'majority',
  serverSelectionTimeoutMS: 10000, // Increased timeout
  socketTimeoutMS: 45000,
  bufferMaxEntries: 0,
  bufferCommands: false,
  maxPoolSize: 10,
  minPoolSize: 1,
  maxIdleTimeMS: 30000,
  connectTimeoutMS: 10000,
  heartbeatFrequencyMS: 10000,
  retryReads: true,
  // Additional SSL options for Atlas
  tls: true,
  tlsAllowInvalidCertificates: true,
  tlsAllowInvalidHostnames: true,
  tlsInsecure: true,
};

// Connect to MongoDB Atlas
const connectDB = async () => {
  try {
    const conn = await mongoose.connect(process.env.MONGODB_URI, mongoOptions);
    
    winston.info(`MongoDB Atlas Connected: ${conn.connection.host}`);
    
    // Handle connection events
    mongoose.connection.on('error', (err) => {
      winston.error('MongoDB connection error:', err);
    });
    
    mongoose.connection.on('disconnected', () => {
      winston.warn('MongoDB disconnected');
    });
    
    mongoose.connection.on('reconnected', () => {
      winston.info('MongoDB reconnected');
    });
    
    return conn;
  } catch (error) {
    winston.error('MongoDB connection failed:', error);
    process.exit(1);
  }
};

// Graceful shutdown
const disconnectDB = async () => {
  try {
    await mongoose.connection.close();
    winston.info('MongoDB connection closed');
  } catch (error) {
    winston.error('Error closing MongoDB connection:', error);
  }
};

module.exports = {
  connectDB,
  disconnectDB,
  mongoOptions,
};
