const mongoose = require('mongoose');
require('dotenv').config();

// Connect to MongoDB with SSL configuration
mongoose.connect(process.env.MONGODB_URI, {
  useNewUrlParser: true,
  useUnifiedTopology: true,
  ssl: true,
  sslValidate: false,
  retryWrites: true,
  w: 'majority',
  serverSelectionTimeoutMS: 5000,
  socketTimeoutMS: 45000,
  bufferMaxEntries: 0,
  bufferCommands: false,
  maxPoolSize: 10,
  minPoolSize: 1,
  maxIdleTimeMS: 30000,
  connectTimeoutMS: 10000,
  heartbeatFrequencyMS: 10000,
  retryReads: true,
})
  .then(() => console.log('Connected to MongoDB Atlas'))
  .catch(err => console.error('MongoDB connection error:', err));

// Import models
const Subscription = require('./src/models/Subscription');
const Organization = require('./src/models/Organization');
const User = require('./src/models/User');

async function checkSubscriptions() {
  try {
    console.log('\n=== CHECKING ALL SUBSCRIPTIONS ===');
    
    // Get all subscriptions
    const subscriptions = await Subscription.find({})
      .populate('organizationId', 'name email _id');
    
    console.log(`\nTotal subscriptions found: ${subscriptions.length}`);
    
    subscriptions.forEach((sub, index) => {
      console.log(`\n${index + 1}. Subscription ID: ${sub._id}`);
      console.log(`   Organization: ${sub.organizationId?.name || 'Unknown'} (${sub.organizationId?._id})`);
      console.log(`   Tier: ${sub.tier}`);
      console.log(`   Status: ${sub.status}`);
      console.log(`   Amount: ${sub.amount} cents (${(sub.amount / 100).toFixed(2)} USD)`);
      console.log(`   Created: ${sub.createdAt}`);
    });

    console.log('\n=== CHECKING ALL ORGANIZATIONS ===');
    
    // Get all organizations
    const organizations = await Organization.find({})
      .populate('staff.userId', 'email role');
    
    console.log(`\nTotal organizations found: ${organizations.length}`);
    
    organizations.forEach((org, index) => {
      console.log(`\n${index + 1}. Organization ID: ${org._id}`);
      console.log(`   Name: ${org.name}`);
      console.log(`   Email: ${org.email}`);
      console.log(`   Staff count: ${org.staff?.length || 0}`);
      
      if (org.staff && org.staff.length > 0) {
        org.staff.forEach((staff, staffIndex) => {
          console.log(`     Staff ${staffIndex + 1}: ${staff.userId?.email || 'Unknown'} (${staff.role || 'No role'})`);
        });
      }
    });

    console.log('\n=== CHECKING ALL USERS ===');
    
    // Get all users
    const users = await User.find({})
      .populate('organizationId', 'name');
    
    console.log(`\nTotal users found: ${users.length}`);
    
    users.forEach((user, index) => {
      console.log(`\n${index + 1}. User ID: ${user._id}`);
      console.log(`   Email: ${user.email}`);
      console.log(`   Role: ${user.role}`);
      console.log(`   Organization: ${user.organizationId?.name || 'None'}`);
    });

  } catch (error) {
    console.error('Error checking subscriptions:', error);
  } finally {
    mongoose.connection.close();
  }
}

checkSubscriptions();

